/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/ResourceAccessor.java,v 1.14 2004/07/30 06:52:27 ozeigermann Exp $
 * $Revision: 1.14 $
 * $Date: 2004/07/30 06:52:27 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import javax.wvcm.AccessControlElement.Privilege;
import javax.wvcm.LockToken;
import javax.wvcm.Principal;
import javax.wvcm.PropertyNameList;
import javax.wvcm.Resource;
import javax.wvcm.SearchToken;
import javax.wvcm.WvcmException;

/**
 * Persistent resource state accessor. There can be various implementations of this
 * interface for server-side as well as for client-side persistent state.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.14 $
 */
public interface ResourceAccessor extends Accessor {
    
    /**
     * Get the resource proxy this accessor is assigned to.
     *
     * @return   the resource proxy this accessor is assigned to.
     */
    public Resource resource();
    
    /**
     * Return a new resource proxy at the location of the associated resource containing
     * the wanted properties. The exact resource type of the resulting resource proxy
     * is determined by this method.
     * The resource content is written to <code>content</code>
     * and <code>content</code> is closed.
     *
     * @param    wantedPropertyList the wanted properties
     * @param    content            an OutputStream
     * @return   a new resource proxy containing the wanted properties
     * @throws   WvcmException
     */
    public Resource doReadContent(PropertyNameList wantedPropertyList, OutputStream content) throws WvcmException;
    
    /**
     * Persists content changes to a resource.
     * If <code>contentIdentifier</code> matches the current
     * state identifier of the persistent resource,
     * the content of the resource is replaced with the
     * bytes read from <code>content</code>, and <code>content</code>
     * is then closed.
     * If reading from the stream throws a <code>java.io.IOException</code>,
     * then no further data will be read from the stream,
     * and after attempting to close the stream, a <code>WvcmException</code>
     * wrapping the <code> IOException</code> will be thrown,
     * possibly leading to incomplete data being stored on the resource.
     * @throws WvcmException if the resource identified by this {@link Resource}
     * does not exist.
     *
     * @param    content the content input stream
     * @param    contentIdentifier the content identifier to match the current identifier
     * @throws   WvcmException
     */
    public void doWriteContent( InputStream content, String contentIdentifier ) throws WvcmException;
    
    /**
     * Return a new resource proxy at the location of the associated resource containing
     * the wanted properties. The exact resource type of the resulting resource proxy
     * is determined by this method.
     *
     * @param    wantedPropertyList the wanted properties
     * @return   a new resource proxy containing the wanted properties
     * @throws   WvcmException
     */
    public Resource doReadProperties( PropertyNameList wantedPropertyList) throws WvcmException;
    
    /**
     * Persists property changes to associated resource.
     *
     * @throws   WvcmException
     */
    public void doWriteProperties() throws WvcmException;
    
    /**
     * Deletes the resource at the location of the associated resource.
     * If a folder is deleted, its members are deleted as well.
     */
    public void doDelete() throws WvcmException;
    
    /**
     * Create a copy of the resource identified by the associated proxy
     * at the location identified by the <code>destination</code>.
     * @param destination The location of the new resource created by doCopy.
     * @param overwrite If <code>false</code> the existence of a resource
     * at the destination will cause the copy to fail; otherwise,
     * doCopy will replace the destination resource.
     */
    public void doCopy(String destination, boolean overwrite) throws WvcmException;
    
    /**
     * Moves the resource identified by associated proxy
     * to the location identified by the <code>destination</code>.
     * @param destination The new location of the resource.
     * @param overwrite If <code>false</code> the existence of a resource
     * at the destination will cause doMove to fail; otherwise,
     * doMove will replace the destination resource.
     */
    public void doMove(String destination, boolean overwrite) throws WvcmException;
    
    /**
     * Checks whether the resource exists at the storage where this accessor
     * is in charge of.
     *
     * @return   true, if the resource exists
     * @throws   WvcmException
     */
    public boolean doCheckExists() throws WvcmException;
    
    /**
     * Return a list of {@link Resource} objects containing the wanted properties
     * according to the conditions of the specified search token from the scope
     * defined by the associated resource.
     * A requested property named XXX can be retrieved from
     * the resource with the <code>getXxx</code> method.
     *
     * @param    wantedPropertyList  the wanted properties
     * @param    searchToken         a  SearchToken
     * @return   the result list
     * @throws   WvcmException
     */
    public List doSearch(PropertyNameList wantedPropertyList, SearchToken searchToken) throws WvcmException;
    
    /**
     * Return the list of {@link AccessControlElement} instances (ACL) defined
     * on the associated resource.
     * The ACL specifies the list of access control elements (ACEs), which define what principals
     * are to get what privileges for this resource.
     * Each ACE specifies the set of privileges to be either granted or denied to a single principal.
     * If the ACL is empty, no principal is granted any privilege.
     *
     * @param    includeInherited    if false, only ACEs defined for the resource are returned;
     *                               otherwise, the ACL includes all inherited ACEs
     * @return   a List
     * @throws   WvcmException
     */
    public List doReadAccessControlList( boolean includeInherited ) throws WvcmException;
    
    /**
     * Modifies the access control list (ACL) of the associated resource. Specifically, this method only
     * permits modification to ACEs that are not inherited, and are not protected.
     *
     * @param    acl                 a list of {@link AccessControlElement} instances
     * @throws   WvcmException
     */
    public void doWriteAccessControlList( List acl ) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Locks this resource.
     *
     * @param    timeout             a  Timeout
     * @param    deep                a  boolean
     * @param    exclusive           a  boolean
     * @param    owner               a  String
     * @return   a LockToken
     * @throws   WvcmException
     */
    public LockToken doLock(LockToken.Timeout timeout, boolean deep, boolean exclusive, String owner) throws WvcmException;
    
    /**
     * NOT YET STANDARD
     * Releases the specified lock of this resource.
     *
     * @param    lockToken           a  LockToken
     * @throws   WvcmException
     */
    public void doUnlock(LockToken lockToken) throws WvcmException;
}

