/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/FolderWebdavAccessor.java,v 1.17 2005/03/04 18:14:31 pnever Exp $
 * $Revision: 1.17 $
 * $Date: 2005/03/04 18:14:31 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav;
import org.apache.wvcm.store.webdav.request.*;

import java.util.Iterator;
import java.util.List;
import javax.wvcm.PropertyNameList;
import javax.wvcm.Resource;
import javax.wvcm.WvcmException;
import javax.wvcm.WvcmException.ReasonCode;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.EntityEnclosingMethod;
import org.apache.webdav.methods.BindMethod;
import org.apache.webdav.methods.PropfindMethod;
import org.apache.webdav.methods.RebindMethod;
import org.apache.webdav.methods.ReportMethod;
import org.apache.webdav.methods.UnbindMethod;
import org.apache.wvcm.LocationImpl;
import org.apache.wvcm.store.FolderAccessor;
import org.apache.wvcm.store.webdav.request.ReportRequest.ExpandPropertyReportRequest;
import org.apache.wvcm.store.webdav.request.ReportRequest.LocateByHistoryReportRequest;
import org.apache.wvcm.store.webdav.response.ErrorResponse;
import org.apache.wvcm.store.webdav.response.MultistatusResponse;
import org.apache.wvcm.store.webdav.response.NoProptatMultistatusResponse;

/**
 * WebDAV-based implementation of FolderResourceAccessor.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.17 $
 */
public class FolderWebdavAccessor extends ResourceWebdavAccessor implements FolderAccessor {
    
    /**
     * Constructor.
     */
    public FolderWebdavAccessor( Resource resource ) {
        super( resource );
    }
    
    /**
     * Return a list of proxies containing the requested set of properties, where the list
     * contains the folder itself and other members of that folder.
     *
     * @param    wantedPropertyList  the requested set of properties
     * @param    deep                if true, the result contains a proxy for every member
     *                               of the folder associated to this accessor; otherwise,
     *                               the result contains only proxies for the bound members
     *                               of the folder.
     * @return   an Iterator
     * @throws   WvcmException
     */
    public Iterator doReadMemberList(PropertyNameList wantedPropertyList, boolean deep) throws WvcmException {
        LocationImpl loc = (LocationImpl)resource().location();
        String resourcepath = loc.path();
        EntityEnclosingMethod method = null;
        AbstractRequest req = null;
        if (containsNestedProperties(wantedPropertyList)) {
            method = new ReportMethod(loc.escapedPath());
            req = new ExpandPropertyReportRequest(wantedPropertyList);
        }
        else {
            method = new PropfindMethod(loc.escapedPath());
            req = new PropfindRequest(wantedPropertyList);
        }
        method.setRequestBody( req.reqBodyAsString() );
        method.addRequestHeader( "Content-Type", "text/xml; charset=\"utf-8\"" );
        
        if( deep )
            method.addRequestHeader( "Depth", "infinity" );
        else
            method.addRequestHeader( "Depth", "1" );
        
        try {
            client().executeMethod( method );
            int sc = method.getStatusCode();
            String st = method.getStatusText();
            switch( sc ) {
                case HttpStatus.SC_MULTI_STATUS:
                    MultistatusResponse resp =
                        new MultistatusResponse( resource(), method.getResponseBodyAsStream(),wantedPropertyList );
                    return resp.createResourceProxies(!deep).iterator();
                case HttpStatus.SC_NOT_FOUND:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.NOT_FOUND, null);
                case HttpStatus.SC_UNAUTHORIZED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_FORBIDDEN:
                case HttpStatus.SC_CONFLICT:
                    ErrorResponse errorRsp = new ErrorResponse( method.getResponseBodyAsStream() );
                    throw new WvcmException(
                        st, resourcepath, errorRsp.getWvcmReasonCode(), null);
                default:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.CANNOT_READ_MEMBERS, null);
            }
        }
        catch (java.io.IOException e) {
            throw new WvcmException(
                "Read failed", resourcepath, ReasonCode.READ_FAILED, new Exception[]{e});
        }
        finally {
            method.releaseConnection();
        }
    }
    
    /**
     * Add a binding.
     *
     * @param    bindingName         a  String
     * @param    boundResource       a  Resource
     * @param    overwrite           a  boolean
     *
     * @throws   WvcmException
     *
     */
    public void doBind(String bindingName, Resource boundResource, boolean overwrite) throws WvcmException {
        LocationImpl loc = (LocationImpl)resource().location();
        String resourcepath = loc.path();
        BindMethod method = new BindMethod(loc.escapedPath());
        BindRequest req = new BindRequest( bindingName, boundResource );
        method.setRequestBody( req.reqBodyAsString() );
        method.addRequestHeader( "Content-Type", "text/xml; charset=\"utf-8\"" );
        method.addRequestHeader( "Overwrite", (overwrite ? "T" : "F") );
        addLockTokens(method);
        
        try {
            method = (BindMethod)retryHandler(req.reqBodyAsString()).executeMethod( method );
            int sc = method.getStatusCode();
            String st = method.getStatusText();
            switch( sc ) {
                case HttpStatus.SC_CREATED:
                case HttpStatus.SC_NO_CONTENT:
                    break;
                case HttpStatus.SC_NOT_FOUND:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_UNAUTHORIZED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_FORBIDDEN:
                case HttpStatus.SC_CONFLICT:
                    ErrorResponse errorRsp = new ErrorResponse( method.getResponseBodyAsStream() );
                    throw new WvcmException(
                        st, resourcepath, errorRsp.getWvcmReasonCode(), null);
                case HttpStatus.SC_LOCKED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.LOCKED, null);
                case HttpStatus.SC_MULTI_STATUS:
                    NoProptatMultistatusResponse resp =
                        new NoProptatMultistatusResponse(resource(), method.getResponseBodyAsStream());
                    List rsL = resp.getResourceStatusList(true);
                    if (rsL.size() > 0) {
                        throw new WvcmException(
                            st, resourcepath, ReasonCode.CONFLICT, getNestedExceptions(rsL));
                    }
                    else {
                        break;
                    }
                default:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.FORBIDDEN, null);
            }
        }
        catch (java.io.IOException e) {
            throw new WvcmException(
                "Read failed", resourcepath, ReasonCode.READ_FAILED, new Exception[]{e});
        }
        finally {
            method.releaseConnection();
        }
    }
    
    /**
     * Remove a binding.
     *
     * @param    bindingName         a  String
     *
     * @throws   WvcmException
     */
    public void doUnbind(String bindingName) throws WvcmException {
        LocationImpl loc = (LocationImpl)resource().location();
        String resourcepath = loc.path();
        UnbindMethod method = new UnbindMethod(loc.escapedPath());
        UnbindRequest req = new UnbindRequest( bindingName );
        method.setRequestBody( req.reqBodyAsString() );
        method.addRequestHeader( "Content-Type", "text/xml; charset=\"utf-8\"" );
        addLockTokens(method);
        
        try {
            method = (UnbindMethod)retryHandler(req.reqBodyAsString()).executeMethod( method );
            int sc = method.getStatusCode();
            String st = method.getStatusText();
            switch( sc ) {
                case HttpStatus.SC_OK:
                    break;
                case HttpStatus.SC_NOT_FOUND:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_UNAUTHORIZED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_FORBIDDEN:
                case HttpStatus.SC_CONFLICT:
                    ErrorResponse errorRsp = new ErrorResponse( method.getResponseBodyAsStream() );
                    throw new WvcmException(
                        st, resourcepath, errorRsp.getWvcmReasonCode(), null);
                case HttpStatus.SC_LOCKED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.LOCKED, null);
                case HttpStatus.SC_MULTI_STATUS:
                    NoProptatMultistatusResponse resp =
                        new NoProptatMultistatusResponse(resource(), method.getResponseBodyAsStream());
                    List rsL = resp.getResourceStatusList(true);
                    if (rsL.size() > 0) {
                        throw new WvcmException(
                            st, resourcepath, ReasonCode.CANNOT_UNBIND_RESOURCE, getNestedExceptions(rsL));
                    }
                    else {
                        break;
                    }
                default:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.FORBIDDEN, null);
            }
        }
        catch (java.io.IOException e) {
            throw new WvcmException(
                "Read failed", resourcepath, ReasonCode.READ_FAILED, new Exception[]{e});
        }
        finally {
            method.releaseConnection();
        }
    }
    
    /**
     * Remove a binding and add a binding.
     *
     * @param    bindingName         a  String
     * @param    boundResource       a  Resource
     * @param    overwrite           a  boolean
     *
     * @throws   WvcmException
     *
     */
    public void doRebind(String bindingName, Resource boundResource, boolean overwrite) throws WvcmException {
        LocationImpl loc = (LocationImpl)resource().location();
        String resourcepath = loc.path();
        RebindMethod method = new RebindMethod(loc.escapedPath());
        RebindRequest req = new RebindRequest( bindingName, boundResource );
        method.setRequestBody( req.reqBodyAsString() );
        method.addRequestHeader( "Content-Type", "text/xml; charset=\"utf-8\"" );
        method.addRequestHeader( "Overwrite", (overwrite ? "T" : "F") );
        
        try {
            method = (RebindMethod)retryHandler(req.reqBodyAsString()).executeMethod( method );
            int sc = method.getStatusCode();
            String st = method.getStatusText();
            switch( sc ) {
                case HttpStatus.SC_CREATED:
                case HttpStatus.SC_NO_CONTENT:
                    break;
                case HttpStatus.SC_NOT_FOUND:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_UNAUTHORIZED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                case HttpStatus.SC_FORBIDDEN:
                case HttpStatus.SC_CONFLICT:
                    ErrorResponse errorRsp = new ErrorResponse( method.getResponseBodyAsStream() );
                    throw new WvcmException(
                        st, resourcepath, errorRsp.getWvcmReasonCode(), null);
                case HttpStatus.SC_MULTI_STATUS:
                    NoProptatMultistatusResponse resp =
                        new NoProptatMultistatusResponse(resource(), method.getResponseBodyAsStream());
                    List rsL = resp.getResourceStatusList(true);
                    if (rsL.size() > 0) {
                        throw new WvcmException(
                            st, resourcepath, ReasonCode.CONFLICT, getNestedExceptions(rsL));
                    }
                    else {
                        break;
                    }
                default:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.FORBIDDEN, null);
            }
        }
        catch (java.io.IOException e) {
            throw new WvcmException(
                "Read failed", resourcepath, ReasonCode.READ_FAILED, new Exception[]{e});
        }
        finally {
            method.releaseConnection();
        }
    }
    
    /**
     * Return a list of {@link ControllableResource} objects
     * that identify the controllable resources that are members of
     * this {@link ControllableFolder} and whose versionHistory property identifies
     * a member of the <code>versionHistoryList</code> parameter.
     * @param versionHistoryList A list of {@link VersionHistory} objects that
     * are being located.
     * @param wantedPropertyList The properties to be retrieved for the
     * returned version-controlled resources.
     */
    public List doLocateByHistoryReport(List versionHistoryList, PropertyNameList wantedPropertyList) throws WvcmException {
        LocationImpl loc = (LocationImpl)resource().location();
        String resourcepath = loc.path();
        ReportMethod method = new ReportMethod(loc.escapedPath());
        ReportRequest req = new LocateByHistoryReportRequest( versionHistoryList, wantedPropertyList );
        method.setRequestBody( req.reqBodyAsString() );
        method.addRequestHeader( "Content-Type", "text/xml; charset=\"utf-8\"" );
        method.addRequestHeader( "Depth", "infinity" );
        
        try {
            client().executeMethod( method );
            int sc = method.getStatusCode();
            String st = method.getStatusText();
            switch( sc ) {
                case HttpStatus.SC_MULTI_STATUS:
                    MultistatusResponse resp =
                        new MultistatusResponse( resource(), method.getResponseBodyAsStream(),wantedPropertyList );
                    return resp.createResourceProxies();
                case HttpStatus.SC_FORBIDDEN:
                case HttpStatus.SC_CONFLICT:
                    ErrorResponse errorRsp = new ErrorResponse( method.getResponseBodyAsStream() );
                    throw new WvcmException(
                        st, resourcepath, errorRsp.getWvcmReasonCode(), null);
                case HttpStatus.SC_NOT_FOUND:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.NOT_FOUND, null);
                case HttpStatus.SC_UNAUTHORIZED:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.UNAUTHORIZED, null);
                default:
                    throw new WvcmException(
                        st, resourcepath, ReasonCode.FORBIDDEN, null);
            }
        }
        catch (java.io.IOException e) {
            throw new WvcmException(
                "Read failed", resourcepath, ReasonCode.READ_FAILED, new Exception[]{e});
        }
        finally {
            method.releaseConnection();
        }
    }
}

