/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/WebdavProperties.java,v 1.8 2004/07/30 06:52:28 ozeigermann Exp $
 * $Revision: 1.8 $
 * $Date: 2004/07/30 06:52:28 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.NestedPropertyName;
import javax.wvcm.PropertyNameList.PropertyName;
import org.apache.wvcm.util.PropertyNameLists;

/**
 * Provides convenience methods for the handling of WebDAV properties.
 * Translates property name into WebDAV names and vice versa.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.8 $
 */
public class WebdavProperties {
    
    /**
     * Get the WebDAV name for the specified PropertyName
     *
     * @param    propertyName          the property name
     * @return   the WebDAV name
     */
    public static String webdavNameForPropertyName( PropertyName propertyName ) {
        if (propertyName instanceof AttributeName && "DAV:".equals(((AttributeName)propertyName).getNamespace())) {
            propertyName = PropertyNameLists.propertyName(propertyName.getString());
        }
        if (propertyName instanceof NestedPropertyName)
            propertyName = ((NestedPropertyName)propertyName).getName();
        return (String)wvcm2webdav.get( propertyName );
    }
    
    /**
     * Get the PropertyName string for the specified WebDAV property.
     *
     * @param    webdavName          the WebDAV property name
     * @return   a PropertyName (null if no matching property)
     */
    public static PropertyName propertyNameForWebdavName( String webdavName ) {
        return (PropertyName)webdav2wvcm.get( webdavName );
    }
    
    /**
     * Convert language-tag [RFC2616, Sec 3.10] to Locale
     *
     * @param    languageTag         a  String
     * @return   a Locale
     */
    public static Locale languageTagToLocale( String languageTag ) {
        StringTokenizer t = new StringTokenizer( languageTag, "-" );
        String language = "";
        String country = "";
        String variant = "";
        
        if (t.hasMoreTokens()) {
            language = t.nextToken();
        }
        if (t.hasMoreTokens()) {
            country = t.nextToken();
        }
        if (t.hasMoreTokens()) {
            variant = t.nextToken();
        }
        return new Locale( language, country, variant );
    }
    
    /**
     * Convert Locale to language-tag [RFC2616, Sec 3.10]
     *
     * @param    locale              a  Locale
     * @return   a String
     */
    public static String localeToLanguageTag( Locale locale ) {
        StringBuffer b = new StringBuffer();
        if (locale.getLanguage() != null && locale.getLanguage().length() > 0) {
            b.append(locale.getLanguage());
            if (locale.getCountry() != null && locale.getCountry().length() > 0) {
                b.append("-").append(locale.getCountry());
                if (locale.getVariant() != null && locale.getVariant().length() > 0) {
                    b.append("-").append(locale.getVariant());
                }
            }
        }
        return b.toString();
    }
    
    /**
     * Mapping: WVCM PropertyName -> WebDAV property name (String)
     */
    private static Map wvcm2webdav = new HashMap();
    static{
        wvcm2webdav.put( PropertyName.ACTIVITY_CHECKOUT_LIST, "activity-checkout-set" );
        wvcm2webdav.put( PropertyName.ACTIVITY_FOLDER_LIST, "activity-collection-set" );
        wvcm2webdav.put( PropertyName.ACTIVITY_LIST, "activity-set" );
        wvcm2webdav.put( PropertyName.ACTIVITY_VERSION_LIST, "activity-version-set" );
        wvcm2webdav.put( PropertyName.AUTO_MERGE_LIST, "auto-merge-set" );
        wvcm2webdav.put( PropertyName.BASELINE_CONTROLLED_FOLDER_LIST, "baseline-controlled-collection-set" );
        wvcm2webdav.put( PropertyName.BASELINE_FOLDER, "baseline-collection" );
        wvcm2webdav.put( PropertyName.CHECKED_IN, "checked-in" );
        wvcm2webdav.put( PropertyName.CHECKED_OUT, "checked-out" );
        wvcm2webdav.put( PropertyName.CHECKIN_FORK, "checkin-fork" );
        wvcm2webdav.put( PropertyName.CHECKOUT_FORK, "checkout-fork" );
        wvcm2webdav.put( PropertyName.CHECKOUT_LIST, "checkout-set" );
        wvcm2webdav.put( PropertyName.COMMENT, "comment" );
        wvcm2webdav.put( PropertyName.CONTENT_IDENTIFIER, "getetag" );
        wvcm2webdav.put( PropertyName.CONTENT_LANGUAGE, "getcontentlanguage" );
        wvcm2webdav.put( PropertyName.CONTENT_LENGTH, "getcontentlength" );
        wvcm2webdav.put( PropertyName.CONTENT_TYPE, "getcontenttype" );
        wvcm2webdav.put( PropertyName.CREATION_DATE, "creationdate" );
        wvcm2webdav.put( PropertyName.CREATOR_DISPLAY_NAME, "creator-displayname" );
        wvcm2webdav.put( PropertyName.CURRENT_ACTIVITY_LIST, "current-activity-set" );
        wvcm2webdav.put( PropertyName.CURRENT_WORKSPACE_LIST, "current-workspace-set" );
        wvcm2webdav.put( PropertyName.DISPLAY_NAME, "displayname" );
        wvcm2webdav.put( PropertyName.ECLIPSED_LIST, "eclipsed-set" );
        wvcm2webdav.put( PropertyName.LABEL_NAME_LIST, "label-name-set" );
        wvcm2webdav.put( PropertyName.LAST_MODIFIED, "getlastmodified" );
        wvcm2webdav.put( PropertyName.MERGE_LIST, "merge-set" );
        wvcm2webdav.put( PropertyName.PARENT_BINDING_LIST, "parent-set" );
        wvcm2webdav.put( PropertyName.PREDECESSOR_LIST, "predecessor-set" );
        wvcm2webdav.put( PropertyName.RESOURCE_IDENTIFIER, "resource-id" );
        wvcm2webdav.put( PropertyName.ROOT_FOLDER, "baseline-controlled-collection" );
        wvcm2webdav.put( PropertyName.ROOT_VERSION, "root-version" );
        wvcm2webdav.put( PropertyName.SUBACTIVITY_LIST, "subactivity-set" );
        wvcm2webdav.put( PropertyName.SUBBASELINE_LIST, "subbaseline-set" );
        wvcm2webdav.put( PropertyName.SUCCESSOR_LIST, "successor-set" );
        wvcm2webdav.put( PropertyName.UNRESERVED, "unreserved" );
        wvcm2webdav.put( PropertyName.VERSION_CONTROLLED_CONFIGURATION, "version-controlled-configuration" );
        wvcm2webdav.put( PropertyName.VERSION_HISTORY, "version-history" );
        wvcm2webdav.put( PropertyName.VERSION_HISTORY_FOLDER_LIST, "version-history-collection-set" );
        wvcm2webdav.put( PropertyName.VERSION_NAME, "version-name" );
        wvcm2webdav.put( PropertyName.WORKSPACE, "workspace" );
        wvcm2webdav.put( PropertyName.WORKSPACE_CHECKOUT_LIST, "workspace-checkout-set" );
        wvcm2webdav.put( PropertyName.WORKSPACE_FOLDER_LIST, "workspace-collection-set" );
        
        // NOT YET STANDARD: ACL-related properties
        wvcm2webdav.put( PropertyName.GROUP_MEMBER_LIST, "group-member-set" );
        wvcm2webdav.put( PropertyName.GROUP_MEMBERSHIP, "group-membership" );
        wvcm2webdav.put( PropertyName.OWNER, "owner" );
        wvcm2webdav.put( PropertyName.PRINCIPAL_FOLDER_LIST, "principal-collection-set" );
        wvcm2webdav.put( PropertyName.PRIVILEGE_FOLDER_LIST, "privilege-collection-set" );
        wvcm2webdav.put( PropertyName.SUPPORTED_PRIVILEGE_LIST, "supported-privilege-set" );
        wvcm2webdav.put( PropertyName.MODIFICATION_DATE, "modificationdate" );
        wvcm2webdav.put( PropertyName.CREATION_USER, "creationuser" );
        wvcm2webdav.put( PropertyName.MODIFICATION_USER, "modificationuser" );
    }
    
    /**
     * Mapping: WebDAV property name (String) -> WVCM PropertyName
     */
    private static Map webdav2wvcm = new HashMap();
    static{
        // generate the inverse
        Iterator i = wvcm2webdav.entrySet().iterator();
        while (i.hasNext()) {
            Map.Entry e = (Map.Entry)i.next();
            webdav2wvcm.put(e.getValue(), e.getKey());
        }
    }
}

