/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/request/ReportRequest.java,v 1.4 2004/07/30 06:52:29 ozeigermann Exp $
 * $Revision: 1.4 $
 * $Date: 2004/07/30 06:52:29 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.request;
import java.util.Arrays;
import java.util.List;
import javax.wvcm.AccessControlElement.Privilege;
import javax.wvcm.Principal;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.NestedPropertyName;
import javax.wvcm.PropertyNameList.PropertyName;
import org.apache.wvcm.store.webdav.WebdavProperties;
import org.jdom.Document;
import org.jdom.Element;

/**
 * Handler for REPORT request bodies.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.4 $
 */
public class ReportRequest extends AbstractRequest {
    
    public static class LocateByHistoryReportRequest extends ReportRequest {
        
        /**
         * Constructor
         */
        public LocateByHistoryReportRequest( List versionHistoryList, PropertyNameList pnameList ) {
            Element locatebyhistElm = new Element( "locate-by-history", dnsp );
            Element vhsetElm = new Element( "version-history-set", dnsp );
            locatebyhistElm.addContent( vhsetElm );
            createHrefElements( vhsetElm, versionHistoryList );
            createPropElement( locatebyhistElm, pnameList );
            setReqBodyDoc( new Document(locatebyhistElm) );
        }
    }
    
    public static class ExpandPropertyReportRequest extends ReportRequest {
        
        /**
         * Constructor
         */
        public ExpandPropertyReportRequest( PropertyNameList pnameList ) {
            Element exppropElm = new Element( "expand-property", dnsp );
            createPropertyElements( exppropElm, pnameList );
            setReqBodyDoc( new Document(exppropElm) );
        }
        
        protected void createDefaultPropertyElement(Element contextElm, String name) {
            createPropertyElement(contextElm, name);
        }
        
        /**
         * Create <property> elements in the specified context.
         *
         * @param    contextElm the context
         * @param    pnameList the wanted properties
         *
         */
        protected void createPropertyElements( Element contextElm, PropertyNameList pnameList ) {
            createDefaultPropertyElements(contextElm, pnameList);
            if( pnameList == null || pnameList.getPropertyNames() == null)
                return;
            
            PropertyName[] pnames;
            pnames = pnameList.getPropertyNames();
            for( int i = 0; i < pnames.length; i++ ) {
                Element propElm = null;
                PropertyName pname = pnames[i];
                String webdavname = WebdavProperties.webdavNameForPropertyName( pname );
                if( pname instanceof AttributeName ) {
                    AttributeName attname = (AttributeName)pname;
                    propElm = new Element( "property", dnsp );
                    propElm.setAttribute( "name", attname.getString() );
                    propElm.setAttribute( "namespace", attname.getNamespace() );
                }
                else if( (pname instanceof NestedPropertyName) && webdavname != null ) {
                    NestedPropertyName npname = (NestedPropertyName)pname;
                    propElm = new Element( "property", dnsp );
                    propElm.setAttribute( "name", webdavname );
                    createPropertyElements( propElm, npname.getNested() );
                }
                else if( (pname instanceof NestedPropertyName) && ((NestedPropertyName)pname).getName() instanceof AttributeName  ) {
                    NestedPropertyName npname = (NestedPropertyName)pname;
                    AttributeName attname = (AttributeName)npname.getName();
                    propElm = new Element( "property", dnsp );
                    propElm.setAttribute( "name", attname.getString() );
                    propElm.setAttribute( "namespace", attname.getNamespace() );
                    createPropertyElements( propElm, npname.getNested() );
                }
                else if( webdavname != null ) {
                    propElm = new Element( "property", dnsp );
                    propElm.setAttribute( "name", webdavname );
                }
                else {
                    if( pname == PropertyName.IS_CHECKED_OUT ) {
                        // check whether checked-out is present
                        if( !Arrays.asList(pnames).contains(PropertyName.CHECKED_OUT) ) {
                            propElm = new Element( "property", dnsp );
                            propElm.setAttribute( "name", "checked-out" );
                        }
                    }
                    else if( pname == PropertyName.CONTENT_CHARACTER_SET ) {
                        if( !Arrays.asList(pnames).contains(PropertyName.CONTENT_TYPE) ) {
                            propElm = new Element( "property", dnsp );
                            propElm.setAttribute( "name", "getcontenttype" );
                        }
                    }
                    else if( pname == PropertyName.VERSION_CONTROLLABLE || pname == PropertyName.BASELINE_CONTROLLABLE ) {
                        // check whether VERSION-CONTROL is in the supported-method-set
                        propElm = new Element( "property", dnsp );
                        propElm.setAttribute( "name", "supported-method-set" );
                    }
                    // TODO: prio=h, effort=0.5, descr=(handle PropertyName.ALL_ATTRIBUTES)
                    // TODO: prio=l, effort=2.0, descr=(handle non-WebDAV "dirty-property-list","is-dirty-content","is-stale-content","server-state","stale-property-list")
                }
                
                if( propElm != null )
                    contextElm.addContent( propElm );
            }
        }
        
        /**
         * Add default property needed for resource type determination
         *
         * @param    exppropElm          an Element
         * @param    propname            a  String
         */
        private void createPropertyElement( Element exppropElm, String propname ) {
            Element propElm = new Element( "property", dnsp );
            propElm.setAttribute( "name", propname );
            exppropElm.addContent( propElm );
        }
    }
}

