/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/request/SearchRequest.java,v 1.3 2004/07/30 06:52:29 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:29 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.request;

import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.PropertyName;
import javax.wvcm.SearchToken;
import javax.wvcm.SearchToken.ContOperator;
import javax.wvcm.SearchToken.ContentExpression;
import javax.wvcm.SearchToken.Expression;
import javax.wvcm.SearchToken.LocationFilter;
import javax.wvcm.SearchToken.NestedExpression;
import javax.wvcm.SearchToken.PropOperator;
import javax.wvcm.SearchToken.PropertyExpression;
import javax.wvcm.SearchToken.XPathAttributeName;
import javax.wvcm.WvcmException;
import org.jdom.Attribute;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.Text;

/**
 * Handler for SEARCH request bodies.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.3 $
 */
public class SearchRequest extends AbstractRequest {
    
    private int xpathExpressionCount = 0;
    
    
    private Set getPropertyNamesAsSet( PropertyNameList pnameList ) {
        if (pnameList != null && pnameList.getPropertyNames() != null) {
            return new HashSet(Arrays.asList(pnameList.getPropertyNames()));
        }
        else {
            return new HashSet();
        }
    }
    
    /**
     * Constructor
     */
    public SearchRequest( String resourcePath, PropertyNameList pnameList, SearchToken searchToken ) throws WvcmException {
        Element searchreqElm = new Element( "searchrequest", dnsp );
        Element basicsearchElm = new Element( "basicsearch", dnsp );
        searchreqElm.addContent( basicsearchElm );
        // select
        // add first properties from the sortCriteria to be able to reference them
        if( searchToken != null && searchToken.sortCriteria() != null ) {
            Set pnamesSet = getPropertyNamesAsSet( pnameList );
            for( int i = 0; i < searchToken.sortCriteria().length; i++ ) {
                PropertyName pn = searchToken.sortCriteria()[i].propertyName();
                if (pn instanceof XPathAttributeName) {
                    pnamesSet.add( pn );
                }
            }
            pnameList =
                new PropertyNameList((PropertyName[])pnamesSet.toArray(new PropertyName[pnamesSet.size()]) );
        }
        // create the select element
        Element selectElm = new Element( "select", dnsp );
        createPropElement( selectElm, pnameList );
        basicsearchElm.addContent( selectElm );
        // from
        Element fromElm = new Element( "from", dnsp );
        Element scopeElm = new Element( "scope", dnsp );
        fromElm.addContent( scopeElm );
        createHrefElement( scopeElm, resourcePath );
        basicsearchElm.addContent( fromElm );
        if( searchToken != null ) {
            // depth
            if( searchToken.depth() != null ) {
                Element depthElm = new Element( "depth", dnsp );
                depthElm.addContent( new Text(searchToken.depth().toString()) );
                scopeElm.addContent( depthElm );
            }
            // location filters
            if (searchToken.searchCondition() != null ) {
                createLocationFilters(scopeElm, searchToken.searchCondition().filters());
            }
            // where
            if( searchToken.searchCondition() != null && searchToken.searchCondition().expression() != null) {
                Element whereElm = new Element( "where", dnsp );
                createExpression( whereElm, searchToken.searchCondition().expression() );
                basicsearchElm.addContent( whereElm );
            }
            // orderby
            if( searchToken.sortCriteria() != null ) {
                Element orderbyElm = new Element( "orderby", dnsp );
                for( int i = 0; i < searchToken.sortCriteria().length; i++ ) {
                    Element p = createPropertyElementForOrderBy( searchToken.sortCriteria()[i].propertyName() );
                    if( p != null ) {
                        Element orderElm = new Element( "order", dnsp );
                        Element propElm = new Element( "prop", dnsp );
                        propElm.addContent( p );
                        orderElm.addContent( propElm );
                        if( searchToken.sortCriteria()[i].direction() != null ) {
                            orderElm.addContent( new Element(searchToken.sortCriteria()[i].direction().toString(), dnsp) );
                        }
                        if( searchToken.sortCriteria()[i].caseless() != null ) {
                            orderElm.setAttribute( "caseless", (searchToken.sortCriteria()[i].caseless().booleanValue() ? "yes" : "no") );
                        }
                        orderbyElm.addContent( orderElm );
                    }
                }
                basicsearchElm.addContent( orderbyElm );
            }
            // limit
            if( searchToken.limit() != null ) {
                Element limitElm = new Element( "limit", dnsp );
                Element nresElm = new Element( "nresults", dnsp );
                limitElm.addContent( nresElm );
                nresElm.addContent( new Text(searchToken.limit().toString()) );
                basicsearchElm.addContent( limitElm );
            }
        }
        
        setReqBodyDoc( new Document(searchreqElm) );
    }
    
    /**
     * Create property element for specified PropertyName
     *
     * @param    pname a  PropertyName
     * @return   the property element
     */
    private Element createPropertyElementForOrderBy(PropertyName pname) {
        Element propertyElm = null;
        if( pname instanceof XPathAttributeName ) {
            XPathAttributeName attname = (XPathAttributeName)pname;
            Namespace xsvnsp = Namespace.getNamespace("xsv", "http://namespaces.softwareag.com/tamino/webdav");
            propertyElm = new Element( "xpath", xsvnsp );
            propertyElm.setAttribute( "refname", attname.getString() );
            propertyElm.setAttribute( "namespace", attname.getNamespace() );
        }
        else {
            return super.createPropertyElement( pname );
        }
        return propertyElm;
    }

    private void createLocationFilters(Element contextElm, LocationFilter[] filters) {
        if (filters == null) {
            return;
        }
        for (int i = 0; i < filters.length; i++) {
            LocationFilter f = filters[i];
            if (f.isNegated()) {
                Element exclElm = new Element("exclude-lastpathsegment", dnsp);
                exclElm.addContent(f.pattern());
                contextElm.addContent(exclElm);
            }
            else {
                Element inclElm = new Element("include-lastpathsegment", dnsp);
                inclElm.addContent(f.pattern());
                contextElm.addContent(inclElm);
            }
        }
    }
    
    /**
     * Create the DASL expression
     *
     * @param    contextElm          the context to add the DASL expression
     * @param    expression          the expression
     */
    private void createExpression( Element contextElm, Expression expression ) {
        Element opElm = null;
        if( expression instanceof PropertyExpression ) {
            PropertyExpression propexp = (PropertyExpression)expression;
            Namespace nsp = ( propexp.operator() == PropOperator.CONTAINS
                                 ? Namespace.getNamespace( "s", "http://jakarta.apache.org/slide/")
                                 : dnsp
                            );
            opElm = new Element( propexp.operator().toString(), nsp );
            if( propexp.caseless() != null ) {
                opElm.setAttribute( "caseless", (propexp.caseless().booleanValue() ? "yes" : "no") );
            }
            Element propElm = new Element( "prop", dnsp );
            Element literalElm = new Element( "literal", dnsp );
            propElm.addContent( createPropertyElement(propexp.propertyName()) );
            opElm.addContent( propElm );
            if( propexp.value() instanceof String ) {
                literalElm.addContent( new Text((String)propexp.value()) );
            }
            else if( propexp.value() instanceof Integer ) {
                literalElm.addContent( new Text(((Integer)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Long ) {
                literalElm.addContent( new Text(((Long)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Short ) {
                literalElm.addContent( new Text(((Short)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Byte ) {
                literalElm.addContent( new Text(((Byte)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Character ) {
                literalElm.addContent( new Text(((Character)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Double ) {
                literalElm.addContent( new Text(((Double)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Float ) {
                literalElm.addContent( new Text(((Float)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Locale ) {
                literalElm.addContent( new Text(((Locale)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Date ) {
                literalElm.addContent( new Text(((Date)propexp.value()).toString()) );
            }
            else if( propexp.value() instanceof Boolean ) {
                literalElm.addContent( new Text(((Boolean)propexp.value()).toString()) );
            }
            else {
                // TODO: prio=h, effort=1.0, desc=(handle other data types: Resource, List-of-Resource??)
            }
            opElm.addContent( literalElm );
        }
        else if( expression instanceof ContentExpression ) {
            ContentExpression contexp = (ContentExpression)expression;
            Namespace nsp = null;
            Attribute att = null;
            if( contexp.operator() == ContOperator.XPATH ) {
                nsp = Namespace.getNamespace( "xsv", "http://namespaces.softwareag.com/tamino/webdav");
                xpathExpressionCount++;
                att = new Attribute("name", String.valueOf(xpathExpressionCount));
            }
            else {
                nsp = dnsp;
            }
            opElm = new Element( contexp.operator().toString(), nsp );
            if( att != null )
                opElm.setAttribute(att);
            opElm.addContent( contexp.string() );
        }
        else if( expression instanceof NestedExpression ) {
            NestedExpression nestexp = (NestedExpression)expression;
            opElm = new Element( nestexp.operator().toString(), dnsp );
            if( nestexp.terms() != null ) {
                for( int i = 0; i < nestexp.terms().length; i++ ) {
                    createExpression( opElm, nestexp.terms()[i] );
                }
            }
        }
        if( opElm != null ) {
            if( expression.isNegated() ) {
                Element nopElm = new Element( "not", dnsp );
                nopElm.addContent( opElm );
                contextElm.addContent( nopElm );
            }
            else {
                contextElm.addContent( opElm );
            }
        }
    }
}

