/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/response/AccessControlElementFactory.java,v 1.4 2004/07/30 06:52:30 ozeigermann Exp $
 * $Revision: 1.4 $
 * $Date: 2004/07/30 06:52:30 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.response;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.wvcm.AccessControlElement;
import javax.wvcm.Location;
import javax.wvcm.Principal;
import javax.wvcm.Provider;
import javax.wvcm.WvcmException;
import org.apache.wvcm.LocationImpl;
import org.apache.wvcm.PrincipalImpl;
import org.apache.wvcm.PrivilegeImpl;
import org.apache.wvcm.ProviderImpl;
import org.apache.wvcm.ResourceImpl;
import org.apache.wvcm.util.XPathWrapper;
import org.jdom.Element;
import org.jdom.Namespace;

/**
 * Creates LockToken instances out of XML DAV:lockdiscovery response elements.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.4 $
 */
public class AccessControlElementFactory {

    private static Namespace dnsp = Namespace.getNamespace("d", "DAV:");

    private Provider provider;
    private List accessControlList;
    private Location location;

    public AccessControlElementFactory(Provider provider, Location location) {
        this.provider = provider;
        this.accessControlList = new ArrayList();
        this.location = location;
    }

    public List getAccessControlList() {
        return accessControlList;
    }

    /**
     * Create List of AccessControlElement instances from DAV:acl property included
     * in this multistatus response.
     *
     * @param    includeInherited    a  boolean
     *
     * @return   a List
     *
     * @throws   WvcmException
     *
     */
    public void create(Element aceElm, boolean includeInherited)  throws WvcmException {
        Principal principal = null;
        List privileges = new ArrayList();
        boolean grant = true;

        // get the principal
        Element principalElm = aceElm.getChild("principal", dnsp);
        principal = createPrincipal(principalElm);

        // get the privileges
        Location privilegesLocation = ((ProviderImpl)provider).serverPrivilegesLocation();
        XPathWrapper xp = new XPathWrapper( "d:grant/d:privilege/*", dnsp );
        List privElms = xp.selectNodes(aceElm);
        if (privElms == null || privElms.isEmpty()) {
            xp = new XPathWrapper(
                "d:deny/d:privilege/*", dnsp );
            privElms = xp.selectNodes(aceElm);
            grant = false;
        }
        if (privElms != null) {
            Iterator p = privElms.iterator();
            while (p.hasNext()) {
                Element pElm = (Element)p.next();
                if ("all".equals(pElm.getName())) {
                    privileges.add(AccessControlElement.Privilege.ALL);
                }
                else {
                    LocationImpl l = (LocationImpl)privilegesLocation.child(pElm.getName());
                    privileges.add( l.resource(PrivilegeImpl.class) );
                }
            }
        }

        // create the ACE
        AccessControlElement ace = new AccessControlElement(principal, privileges, grant);

        // check inheritance
        xp = new XPathWrapper( "d:inherited/d:href", dnsp );
        Element hrefElm = (Element)xp.selectSingleNode(aceElm);
        if (hrefElm != null) {
            LocationImpl l = (LocationImpl)provider.location( hrefElm.getText() );
            if (!includeInherited && !this.location.equals(l)) {
                return;   // skip this ACE because it's inherited
            }
            if (!this.location.equals(l)) {
                ace.setInheritedFrom( l.resource(ResourceImpl.class) );
            }
        }

        accessControlList.add( ace );
    }

    private Principal createPrincipal(Element principalElm) throws WvcmException {
        Principal principal = null;
        if (principalElm != null) {
            List pchilds = principalElm.getChildren();
            if (pchilds != null && pchilds.size() == 1 ) {
                Element pchildElm = (Element)pchilds.get(0);
                if ("all".equals(pchildElm.getName())) {
                    principal = Principal.ALL;
                }
                else if ("self".equals(pchildElm.getName())) {
                    principal = Principal.SELF;
                }
                else if ("owner".equals(pchildElm.getName())) {
                    principal = Principal.OWNER;
                }
                else if ("authenticated".equals(pchildElm.getName())) {
                    principal = Principal.AUTHENTICATED;
                }
                else if ("unauthenticated".equals(pchildElm.getName())) {
                    principal = Principal.UNAUTHENTICATED;
                }
                else if ("href".equals(pchildElm.getName())) {
                    LocationImpl l = (LocationImpl)provider.location(pchildElm.getTextTrim() );
                    principal = (Principal)l.resource(PrincipalImpl.class);
                }
            }
        }
        return principal;
    }
}


