/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/response/MultistatusResponse.java,v 1.4 2004/07/30 06:52:30 ozeigermann Exp $
 * $Revision: 1.4 $
 * $Date: 2004/07/30 06:52:30 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.response;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.wvcm.PropertyNameList;
import javax.wvcm.Resource;
import javax.wvcm.WvcmException;
import javax.wvcm.WvcmException.ReasonCode;
import org.apache.wvcm.ResourceImpl;
import org.apache.wvcm.util.XPathWrapper;
import org.jdom.Element;

/**
 * Handler for multistatus response bodies.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.4 $
 */
public class MultistatusResponse extends AbstractResponse {

    private ResourceProxyFactory proxyFactory;
    private AccessControlElementFactory aceFactory;
    private PropertyNameList wantedPropertyList;

    /**
     * Create handler from given input stream.
     */
    public MultistatusResponse(Resource resource, InputStream respBodyAsStream) throws WvcmException {
        this(resource, respBodyAsStream, null);
    }

    /**
     * Create handler from given input stream.
     */
    public MultistatusResponse(Resource resource, InputStream respBodyAsStream, PropertyNameList wantedPropertyList) throws WvcmException {
        super(resource, respBodyAsStream);
        try {
            this.wantedPropertyList = wantedPropertyList;
            this.proxyFactory = new ResourceProxyFactory(requestResource, provider, wantedPropertyList);
            this.aceFactory = new AccessControlElementFactory(provider, resource.location());
        }
        catch( Exception e ) {
            throw new WvcmException( "Could not parse server response",
                                    resource.location().string(), ReasonCode.READ_FAILED, new Exception[]{e} );
        }
    }

    /**
     * Create the resource proxy resulting from the 1st response of this multistatus response.
     *
     * @return   the resulting resource proxy
     * @throws   WvcmException
     */
    public Resource createResourceProxy() throws WvcmException {
        List proxies = createResourceProxies();
        if( proxies != null && proxies.size() > 0 )
            return (Resource)proxies.get(0);
        else
            return null;
    }

    /**
     * Create the resource proxies resulting from this multistatus response.
     *
     * @return   the resulting resource proxies
     * @throws   WvcmException
     */
    public List createResourceProxies() throws WvcmException {
        return createResourceProxies( null, false );
    }

    /**
     * Create the resource proxies resulting from this multistatus response.
     *
     * @param    excludeRequestResource    if true, no proxy is returned for the requestResource itself
     * @return   the resulting resource proxies
     * @throws   WvcmException
     */
    public List createResourceProxies( boolean excludeRequestResource ) throws WvcmException {
        return createResourceProxies( null, excludeRequestResource );
    }

    /**
     * Create List of AccessControlElement instances from DAV:acl property included
     * in this multistatus response.
     *
     * @param    includeInherited    a  boolean
     *
     * @return   a List
     *
     * @throws   WvcmException
     *
     */
    public List createAccessControlList( boolean includeInherited )  throws WvcmException {
        List result = new ArrayList();
        Element ms = respBodyDoc.getRootElement();
        XPathWrapper xp = new XPathWrapper(
            "d:response/d:propstat[contains(string(d:status),\"200\")]/d:prop/d:acl/*", dnsp );

        Iterator i = xp.selectNodes(ms).iterator();
        while (i.hasNext()) {
            Element aceElm = (Element)i.next();
            aceFactory.create(aceElm, includeInherited);
        }
        return aceFactory.getAccessControlList();
    }

    /**
     * Create the resource proxies resulting from this multistatus response.
     *
     * @param    c    the class of which the returned proxies are instances of
     * @param    excludeRequestResource    if true, no proxy is returned for the requestResource itself
     * @return   the resulting proxies (instances of specified class)
     * @throws   WvcmException
     */
    private List createResourceProxies( Class cls, boolean excludeRequestResource ) throws WvcmException {
        List result = new ArrayList();
        Element ms = respBodyDoc.getRootElement();
        XPathWrapper xp = new XPathWrapper( "d:response", dnsp );

        Iterator i = xp.selectNodes(ms).iterator();
        // iterate over <response> element list
        while( i.hasNext() ) {
            Element re = (Element)i.next();
            if( excludeRequestResource && proxyFactory.getLocation(re).equals(requestResource.location()) ) {
                // skip self
                continue;
            }
            result.add( proxyFactory.create(re, cls) );
        }
        return result;
    }

    /**
     * Create the resource proxy resulting from the 1st response of this multistatus response.
     *
     * @return   the resulting resource proxy
     * @throws   WvcmException
     */
    public Resource createAllAttributesProxy() throws WvcmException {
        Element ms = respBodyDoc.getRootElement();
        XPathWrapper xp = new XPathWrapper( "d:response", dnsp );

        Element re = (Element)xp.selectSingleNode(ms);
        return proxyFactory.createAllAttributes(re);
    }
}




