/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/response/ProppatchMultistatusResponse.java,v 1.3 2004/07/30 06:52:31 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:31 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.response;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.PropertyName;
import javax.wvcm.Provider;
import javax.wvcm.Resource;
import javax.wvcm.WvcmException;
import javax.wvcm.WvcmException.ReasonCode;
import org.apache.wvcm.LocationImpl;
import org.apache.wvcm.ProviderImpl;
import org.apache.wvcm.store.webdav.WebdavProperties;
import org.apache.wvcm.util.XPathWrapper;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;

/**
 * Handler for multistatus response bodies from PROPPATCH.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.3 $
 */
public class ProppatchMultistatusResponse {
    
    private static Namespace dnsp = Namespace.getNamespace("d", "DAV:");
    
    private Document respBodyDoc;
    private Resource requestResource;
    private Provider provider;
    
    /**
     * Create handler from given input stream.
     */
    public ProppatchMultistatusResponse( Resource resource, InputStream respBodyAsStream ) throws WvcmException {
        try {
            this.requestResource = resource;
            this.provider = ((LocationImpl)requestResource.location()).provider();
            this.respBodyDoc = ProviderImpl.getSAXBuilder().build( respBodyAsStream );
        }
        catch( Exception e ) {
            throw new WvcmException( "Could not parse server response",
                                    resource.location().string(), ReasonCode.READ_FAILED, new Exception[]{e} );
        }
    }
    
    public List getPropstatList( boolean failedOnly ) throws WvcmException {
        List result = new ArrayList();
        Element ms = respBodyDoc.getRootElement();
        XPathWrapper xp = new XPathWrapper( "d:response", dnsp );
        Element respElm = (Element)xp.selectSingleNode(ms);
        
        xp = new XPathWrapper( "d:propstat", dnsp );
        
        Iterator i = xp.selectNodes(respElm).iterator();
        while( i.hasNext() ) {
            // fetch next propstat
            Element propstatElm = (Element)i.next();
            Element propElm = propstatElm.getChild("prop", dnsp);
            String statusStr = propstatElm.getChild("status", dnsp).getText().trim();
            String message = "";
            Element respdescElm = propstatElm.getChild("responsedescription", dnsp);
            if (respdescElm != null) {
                message = respdescElm.getText().trim();
            }
            
            PropertyName propname =
                WebdavProperties.propertyNameForWebdavName( propElm.getName() );
            if( propname == null ) {
                Namespace ns = propstatElm.getNamespace();
                // dead property to be converted into AttributeName
                propname = new AttributeName( ns.getURI(), propstatElm.getName() );
            }
            
            if (failedOnly && (statusStr.indexOf("200") >= 0)) {
                continue;
            }
            
            int status = -1;
            try {
                StringTokenizer t = new StringTokenizer(statusStr, " ");
                t.nextToken(); // skip HTTP/1.1
                status = Integer.parseInt(t.nextToken());
                StringBuffer b = new StringBuffer();
                while (t.hasMoreTokens()) {
                    b.append(t.nextToken()).append(" ");
                }
                if (message.length() > 0) {
                    b.append("- ").append(message);
                }
                result.add( new Propstat(propname, status, b.toString().trim()) );
            }
            catch (Exception e) {
                // ignore silently
            }
        }
        
        return result;
    }
    
    public static class Propstat {
        private PropertyName propertyName;
        private int status;
        private String message;
        
        public Propstat(PropertyName propertyName, int status, String message) {
            this.propertyName = propertyName;
            this.status = status;
            this.message = message;
        }
        
        public PropertyName getPropertyName() {
            return propertyName;
        }
        
        public int getStatus() {
            return status;
        }
        
        public String getMessage() {
            return message;
        }
    }
}


