/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/src/org/apache/wvcm/store/webdav/response/ResourceProxyFactory.java,v 1.6 2004/07/30 06:52:31 ozeigermann Exp $
 * $Revision: 1.6 $
 * $Date: 2004/07/30 06:52:31 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.store.webdav.response;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.wvcm.Location;
import javax.wvcm.LockToken;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.NestedPropertyName;
import javax.wvcm.PropertyNameList.PropertyName;
import javax.wvcm.Provider;
import javax.wvcm.ProviderFactory;
import javax.wvcm.Resource;
import javax.wvcm.WvcmException;
import org.apache.wvcm.ActivityImpl;
import org.apache.wvcm.BaselineImpl;
import org.apache.wvcm.ControllableFolderImpl;
import org.apache.wvcm.ControllableResourceImpl;
import org.apache.wvcm.FolderVersionImpl;
import org.apache.wvcm.LocationImpl;
import org.apache.wvcm.LockTokenImpl;
import org.apache.wvcm.PrincipalImpl;
import org.apache.wvcm.PrivilegeImpl;
import org.apache.wvcm.ProviderImpl;
import org.apache.wvcm.ResourceImpl;
import org.apache.wvcm.VersionHistoryImpl;
import org.apache.wvcm.VersionImpl;
import org.apache.wvcm.WorkspaceImpl;
import org.apache.wvcm.store.webdav.ResourceWebdavAccessor;
import org.apache.wvcm.util.PropertyNameLists;
import org.apache.wvcm.util.XPathWrapper;
import org.jdom.Element;
import org.jdom.Namespace;

/**
 * Creates resource proxies out of XML DAV:multistatus response elements.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.6 $
 */
public class ResourceProxyFactory {
    
    private static Namespace dnsp = Namespace.getNamespace("d", "DAV:");
    
    private Resource requestResource;
    private Provider provider;
    private PropertyNameList wantedPropertyList;
    
    /**
     * Constructor
     *
     * @param    requestResource     a  Resource
     * @param    provider            a  Provider
     * @param    wantedPropertyList  a  PropertyNameList
     *
     */
    public ResourceProxyFactory(Resource requestResource, Provider provider, PropertyNameList wantedPropertyList) {
        this.requestResource = requestResource;
        this.provider = provider;
        this.wantedPropertyList = wantedPropertyList;
    }
    
    /**
     * Recursive method. Create resource proxy for the specified <response>
     * element. The result will be an instance of the specified type (cls).
     * If (cls == null), the method tries to determine the exact resource type
     * from selected WebDAV properties.
     *
     * @param    re                  an Element
     * @param    cls                 a  Class
     * @return   a Resource
     * @throws   WvcmException
     */
    public Resource create( Element responseElm, Class resourceType ) throws WvcmException {
        Map propertiesContainer = new HashMap();
        Map webdavProperties = new HashMap();
        
        // build the resource if not there
        if (requestResource == null) {
            LocationImpl loc = getLocation(responseElm);
            Class rt = resourceType;
            if (rt == null) {
                rt = ResourceImpl.class;
            }
            requestResource = (ResourceImpl)loc.resource(rt);
        }
        
        PropertiesFactory propFactory =
            new PropertiesFactory(requestResource, provider, wantedPropertyList, propertiesContainer);
        
        // init the container with MISSING_PROPERTY_VALUE
        int numberOfWantedProps = 0;
        if (wantedPropertyList != null && wantedPropertyList.getPropertyNames() != null) {
            numberOfWantedProps = wantedPropertyList.getPropertyNames().length;
            for (int k = 0; k < numberOfWantedProps; k++) {
                PropertyName pname = wantedPropertyList.getPropertyNames()[k];
                if (pname == PropertyName.ALL_ATTRIBUTES) {
                    continue;
                }
                String namespace = "DAV:";
                if (pname instanceof NestedPropertyName) {
                    pname = ((NestedPropertyName)pname).getName();
                }
                if (pname instanceof AttributeName) {
                    namespace = ((AttributeName)pname).getNamespace();
                }
                PropertyName key = null;
                if ("DAV:".equals(namespace)) {
                    key = PropertyNameLists.propertyName(pname.getString());
                }
                if (key == null) {
                    key = new AttributeName(namespace, pname.getString());
                }
                propertiesContainer.put(key, ResourceImpl.MISSING_PROPERTY_VALUE);
            }
        }
        
        // process ALL_ATTRIBUTES first
        ResourceWebdavAccessor accessor = new ResourceWebdavAccessor(requestResource);
        ResourceImpl allAtts = accessor.doReadAllAttributes(wantedPropertyList);
        if (allAtts != null) {
            propertiesContainer.putAll(allAtts.getPropertyContainer());
        }
        
        // get the properties
        XPathWrapper xp;
        // get the properties
        xp = new XPathWrapper(
            "d:propstat[contains(string(d:status),\"200\")]/d:prop/*", dnsp );
        
        Iterator i = xp.selectNodes(responseElm).iterator();
        while( i.hasNext() ) {
            // fetch next property
            Element propertyElm = (Element)i.next();
            webdavProperties.put( propertyElm.getName(), propertyElm );
            propFactory.create(propertyElm);
            // TODO: prio=l, effort=2.0, descr=(handle non-WebDAV "dirty-property-list","is-dirty-content","is-stale-content","server-state","stale-property-list")
        }
        
        // special handling for non-WebDAV properties
        if( webdavProperties.get("supported-method-set") != null ) {
            Element supportedmethods = (Element)webdavProperties.get("supported-method-set");
            if (propertiesContainer.containsKey(PropertyName.VERSION_CONTROLLABLE)) {
                xp = new XPathWrapper( "d:supported-method[@name=\"VERSION-CONTROL\"]", dnsp );
                boolean versioncontrollable = (xp.selectNodes(supportedmethods).size() > 0);
                propertiesContainer.put( PropertyName.VERSION_CONTROLLABLE, new Boolean(versioncontrollable) );
            }
            if (propertiesContainer.containsKey(PropertyName.BASELINE_CONTROLLABLE)) {
                xp = new XPathWrapper( "d:supported-method[@name=\"BASELINE-CONTROL\"]", dnsp );
                boolean baselinecontrollable = (xp.selectNodes(supportedmethods).size() > 0);
                propertiesContainer.put( PropertyName.BASELINE_CONTROLLABLE, new Boolean(baselinecontrollable) );
            }
        }
        
        // create the proxy
        LocationImpl loc = getLocation(responseElm);
        if (resourceType == null || isSubtypeOfPrincipalImpl(resourceType))
            resourceType = determineResourceType(loc, webdavProperties);
        ResourceImpl result = (ResourceImpl)loc.resource( resourceType );
        result.setPropertyContainer( propertiesContainer );
        
        // add the lock tokens
        if (webdavProperties.get("lockdiscovery") != null) {
            LockTokenFactory locktokenFactory = new LockTokenFactory(requestResource, provider);
            Element ldElm = (Element)webdavProperties.get("lockdiscovery");
            Iterator activeLocks = ldElm.getChildren("activelock", dnsp).iterator();
            while (activeLocks.hasNext()) {
                Element alElm = (Element)activeLocks.next();
                LockToken lockToken = locktokenFactory.create(alElm);
                setActiveIfOwner(lockToken);
                result.addLockToken(lockToken);
            }
        }
        return result;
    }
    
    /**
     * Method create
     *
     * @param    propertiesContainer a  Map
     * @param    resourceType        a  Class
     *
     * @return   a Resource
     *
     * @throws   WvcmException
     *
     */
    public Resource create(Location location, Class resourceType, Map propertiesContainer) throws WvcmException {
        ResourceImpl result = (ResourceImpl)((LocationImpl)location).resource( resourceType );
        if (propertiesContainer != null) {
            result.setPropertyContainer( propertiesContainer );
        }
        return result;
    }
    
    /**
     * Method isSubtypeOfPrincipalImpl
     *
     * @param    cls                 a  Class
     *
     * @return   a boolean
     *
     */
    private boolean isSubtypeOfPrincipalImpl( Class cls ) {
        if (cls == PrincipalImpl.class)
            return true;
        else if (cls == null || cls == Object.class)
            return false;
        else
            return isSubtypeOfPrincipalImpl( cls.getSuperclass() );
    }
    
    /**
     *
     */
    public Resource createAllAttributes(Element responseElm) throws WvcmException {
        Map propertiesContainer = new HashMap();
        
        // build the resource if not there
        if (requestResource == null) {
            LocationImpl loc = getLocation(responseElm);
            requestResource = (ResourceImpl)loc.resource(ResourceImpl.class);
        }
        
        PropertiesFactory propFactory =
            new PropertiesFactory(requestResource, provider, wantedPropertyList, propertiesContainer);
        
        // get the properties
        XPathWrapper xp;
        // get the properties
        xp = new XPathWrapper(
            "d:propstat[contains(string(d:status),\"200\")]/d:prop/*", dnsp );
        
        Iterator i = xp.selectNodes(responseElm).iterator();
        while( i.hasNext() ) {
            // fetch next property
            Element propertyElm = (Element)i.next();
            propFactory.create(propertyElm);
        }
        
        // create the proxy
        LocationImpl loc = getLocation(responseElm);
        ResourceImpl result = (ResourceImpl)loc.resource(ResourceImpl.class);
        result.setPropertyContainer( propertiesContainer );
        return result;
    }
    
    /**
     * Method setActiveIfOwner
     *
     * @param    lockToken           a  LockToken
     *
     */
    private void setActiveIfOwner(LockToken lockToken) {
        ProviderImpl p = (ProviderImpl)provider;
        String realm = (String)p.initParameter("realm");
        ProviderFactory.Callback cb = p.callback();
        String user = null;
        if (cb != null) {
            ProviderFactory.Callback.Authentication authentication =
                cb.getAuthentication(realm, new Integer(0));
            user = authentication.loginName();
        }
        boolean isOwner =
            lockToken.getOwner().location().lastSegment().equals("unauthenticated") ||
            lockToken.getOwner().location().lastSegment().equals(user);
        ((LockTokenImpl)lockToken).setActive(isOwner);
    }
    
    /**
     * Get the Location associated to the specified response element (retrieves href element).
     *
     * @param    re the response element
     * @return   the associated Location
     * @throws   WvcmException
     *
     */
    public LocationImpl getLocation( Element re ) throws WvcmException {
        XPathWrapper xp = new XPathWrapper( "string(d:href)", dnsp );
        return new LocationImpl((String)xp.selectNodes(re).get(0), (ProviderImpl)provider, true);
    }
    
    /**
     * Determine the resource type from the specified set of selected WebDAV properties.
     *
     * @param    webdavProperties    a  Map
     * @return   a Class
     * @throws   WvcmException
     */
    private Class determineResourceType( Location loc, Map webdavProperties ) throws WvcmException {
        if (webdavProperties == null || webdavProperties.isEmpty()) {
            return ResourceImpl.class;
        }
        
        Element rt;
        XPathWrapper xp;
        rt = (Element)webdavProperties.get( "resourcetype" );
        List rtnames = new ArrayList();
        if (rt != null) {
            Iterator rti = rt.getChildren().iterator();
            while (rti.hasNext()) {
                rtnames.add( ((Element)rti.next()).getName() );
            }
        }
        
        if( webdavProperties.get("version-controlled-binding-set") != null ) {
            return FolderVersionImpl.class;
        }
        else if( webdavProperties.get("baseline-collection") != null ) {
            return BaselineImpl.class;
        }
        else if( webdavProperties.get("version-name") != null ) {
            return VersionImpl.class;
        }
        else if( webdavProperties.get("workspace-checkout-set") != null ) {
            return WorkspaceImpl.class;
        }
        else {
            // check resourcetype
            if (rtnames.contains("version-history")) {
                return VersionHistoryImpl.class;
            }
            else if (rtnames.contains("principal")) {
                if (webdavProperties.get("group-member-set") != null) {
                    if (rtnames.contains("role")) {
                        return PrincipalImpl.RoleImpl.class;
                    }
                    else {
                        return PrincipalImpl.GroupImpl.class;
                    }
                }
                else {
                    return PrincipalImpl.class;
                }
            }
            else if (rtnames.contains("activity")) {
                return ActivityImpl.class;
            }
                // TODO:
                // TODO: temporary hack!!
                // TODO:
            else if (loc.string().indexOf("/administration/security/actions") > -1) {
                return PrivilegeImpl.class;
            }
                //
                //
                //
            else if (rtnames.contains("collection")) {
                return ControllableFolderImpl.class;
            }
            else {
                return ControllableResourceImpl.class;
            }
        }
    }
}



