/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/test/junit/src/org/apache/wvcm/test/AclTest.java,v 1.3 2004/07/30 06:52:32 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:32 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.test;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.wvcm.AccessControlElement;
import javax.wvcm.AccessControlElement.Privilege;
import javax.wvcm.ControllableResource;
import javax.wvcm.Principal;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.PropertyName;
import javax.wvcm.WvcmException;
import javax.wvcm.WvcmException.ReasonCode;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.apache.wvcm.test.common.AbstractTest;

/**
 * Test ACL related functionality provided by the WVCM implementation.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.3 $
 */
public class AclTest extends AbstractTest {
	
	private Logger logger = Logger.getLogger(this.getClass().getName());
    
    private String privilegesFolderPath = null;
	
	private PropertyNameList ownerProp =
		new PropertyNameList( new PropertyName[]{
				PropertyName.OWNER} );
    
    /**
	 * Default Constructor
	 *
	 * @param    testName            Name of the test to run.
	 *
	 * @throws   Exception
	 *
	 */
    public AclTest(String testName) throws Exception {
		super(testName);
		folderPath = rootFolderPath() + "/foo";
		resourcePath = rootFolderPath() + "/test.xml";
		privilegesFolderPath = contextPath() + "/" + privilegesFolderName;
    }
    
    /**
	 * Method suite
	 *
	 * @return   a Test
	 */
    public static Test suite() {
		return new TestSuite(AclTest.class);
    }
    
    /**
	 * Start test suite contained in this class using the text interface. If a
	 * parameter is passed, it will use it as a test name and run only that test
	 * case.
	 *
	 * @param args   commandline options (first argument = test method name to run)
	 *
	 * @throws Exception
	 */
    public static void main( String[] args ) throws Exception{
		if (args.length == 1) {
			TestSuite testSuite = new TestSuite();
			testSuite.addTest(new AclTest(args[0]));
			junit.textui.TestRunner.run(testSuite);
		}
		else {
			junit.textui.TestRunner.run( suite() );
		}
    }
    
    /**
	 * Common test setup.
	 *
	 * @throws Exception
	 */
    protected void setUp() throws Exception {
		super.setUp ();
    }
    
    /**
	 * Tears down the fixture, for example, close a network connection.
	 * This method is called after a test is executed.
	 *
	 * @throws Exception
	 */
    protected void tearDown() throws Exception {
    }
    
	/**
	 * <ul>
	 * <li>Create a resource, two Principals for the test, and two privileges.
	 * <li>Create two ACE with the created privileges.
	 * <li>Create an ACL with the created ACE.
	 * <li>Write the ACL to the resource.
	 * <li>Read the ACL from the resource and check it is the expected.
	 *
	 * @throws   Exception
	 *
	 */
    public void testDoReadWriteAccessControlList() throws Exception {
		String privWriteName = "/testWrite";
		String privReadName = "/testRead";
		String privilegeWritePath = privilegesFolderPath.concat(privWriteName);
		String privilegeReadPath = privilegesFolderPath.concat(privReadName);
		String testUser1Name = "testUser1";
		String testUser2Name = "testUser2";
		// create resource
		location = provider.location(resourcePath);
		resource = createControllableResource(location);
		
		// create principals
		Principal user1 = createUser(testUser1Name, true);
		Principal user2 = createUser(testUser2Name, true);
		
		// create privileges
		location = provider.location(privilegeWritePath);
		Privilege testWritePriv = createPrivilege(location);
		location = provider.location(privilegeReadPath);
		Privilege testReadPriv = createPrivilege(location);
		
		List acl = new ArrayList();
		List listReadPriv = new ArrayList();
		List listWritePriv = new ArrayList();
		listReadPriv.add(testReadPriv);
		listWritePriv.add(testWritePriv);
		
		AccessControlElement ace1 = new AccessControlElement(user1, listReadPriv, true);
		AccessControlElement ace2 = new AccessControlElement(user2, listWritePriv, false);
		
		acl.add(ace1);
		acl.add(ace2);
		
		resource.doWriteAccessControlList( acl );
		
		List readACL = resource.doReadAccessControlList(true);
		assertNotNull("ACL List read is unexpectedly null.", readACL);
		List aclReceived;
		String tempPrivLocation;
		for (int i=0; i < readACL.size(); i++) {
			aclReceived = ((AccessControlElement)readACL.get(i)).getPrivileges();
			for (int x = 0; x < aclReceived.size(); x++) {
				tempPrivLocation = ((Privilege)aclReceived.get(x)).location().string();
				assertTrue("Unexpected Privilege received.",
						   tempPrivLocation.endsWith("all") ||
						   tempPrivLocation.endsWith(privWriteName) ||
						   tempPrivLocation.endsWith(privReadName));
			}
		}
		
		// Remove the acl from the resource to be able to remove stuff later.
		acl = new ArrayList();
		resource.doWriteAccessControlList(acl);
    }
	
	/**
	 * <ul>
	 * <li>Create a resource (using default user)
	 * <li>Get the owner of the resource.
	 * <li>Create another provider with another user/password (should be allowed
	 * in server).
	 * <li>Create a resource using that provider.
	 * <li>Get the owner of the created resource.
	 *
	 * @throws   Exception
	 *
	 */
	public void testGetOwner() throws Exception{
		// create resource
		location = provider.location(resourcePath);
		resource = createControllableResource(location);
		
		try {
			resource = (ControllableResource)resource.readProperties(ownerProp);
			assertNotNull("Resource is null after reading Owner property.", resource);
		}
		catch (WvcmException unexpected) {
			if (logger.isLoggable(Level.CONFIG))
				logger.config(unexpected.getMessage());
			fail("Unexpected exception when getting owner property.");
		}
		
		Principal owner = resource.getOwner();
		System.out.println(owner.toString());
		assertEquals("Unexpected Owner of created resource.",
					 "expect Error",
					 owner.toString());
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default write.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeWrite() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/write"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege write should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege write should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default read-current-user-privilege-set.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeReadCurrentUserPrivilegeSet() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/read-current-user-privilege-set"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege read-current-user-privilege-set should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege read-current-user-privilege-set should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default unbind.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeUnbind() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/unbind"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege unbind should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege unbind should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default unlock.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeUnlock() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/unlock"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege unlock should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege unlock should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default write-protecte.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeWriteProperties() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/write-properties"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege write-properties should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege write-properties should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default write-content.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeWriteContent() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/write-content"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege write-content should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege write-content should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default read.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeRead() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/read"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege read should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege read should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default bind.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeBind() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/bind"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege bind should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege bind should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default read-acl.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeReadAcl() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/read-acl"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege read-acl should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege read-acl should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a privilege with the same name as the default write-acl.
	 * <li>Check the exception and see it is forbidden.
	 *
	 * @throws Exception
	 */
	public void testUnbindCreateDefaultPrivilegeWriteAcl() throws Exception {
		location = provider.location(privilegesFolderPath.concat("/write-acl"));
		Privilege priv = location.privilege();
		
		try {
			priv.doUnbind();
			fail("Deleting privilege write-acl should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.FORBIDDEN,
						 expected.getReasonCode());
		}
		
		try {
			priv.doCreateResource();
			fail("Overwriting privilege write-acl should have thrown an Exception.");
		}
		catch (WvcmException expected) {
			assertEquals("Unexpected Reason in the thrown exception.",
						 ReasonCode.CANNOT_INITIALIZE_RESOURCE,
						 expected.getReasonCode());
		}
	}
	
	// TODO: ADD MISSING TEST CASES.
}


