/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/test/junit/src/org/apache/wvcm/test/ResourceTest.java,v 1.3 2004/07/30 06:52:33 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:33 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.wvcm.test;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.wvcm.ControllableFolder;
import javax.wvcm.Folder;
import javax.wvcm.Location;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.PropertyName;
import javax.wvcm.Resource;
import javax.wvcm.WvcmException;
import javax.wvcm.WvcmException.ReasonCode;
import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.apache.wvcm.test.common.AbstractTest;
import org.apache.wvcm.test.common.util.ReasonCodeChecker;

/**
 * Functional tests for Resource.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.3 $
 */
public class ResourceTest extends AbstractTest
{
	
	private static Logger logger = Logger.getLogger("org.apache.wvcm.test.ResourceTest");
    
    private String destinationPath = null;
    
	private String testResx1 = "test.xml";
	private String testResxCopy = "testCopy.xml";
	private String testHelloResx = "Hello.txt";
	
    /**
	 * Default Constructor
	 *
	 * @param    testName            Name of the test to run.
	 *
	 * @throws   Exception
	 *
	 */
    public ResourceTest(String testName) throws Exception
	{
		super(testName);
		resourcePath = rootFolderPath() + "/" + testResx1;
		destinationPath = rootFolderPath() + "/" + testResxCopy;
		
    }
    
    /**
	 * Creates a TestSuite with tests contained in this class.
	 *
	 * @return   a Test
	 */
    public static Test suite()
	{
		return new TestSuite(ResourceTest.class);
    }
    
    /**
	 * Start test suite contained in this class using the text interface. If a
	 * parameter is passed, it will use it as a test name and run only that test
	 * case.
	 *
	 * @param args   commandline options (first argument = test method name to run)
	 *
	 * @throws Exception
	 */
    public static void main( String[] args ) throws Exception
	{
		if (args.length == 1)
		{
			TestSuite testSuite = new TestSuite();
			testSuite.addTest(new ResourceTest(args[0]));
			junit.textui.TestRunner.run(testSuite);
		}
		else
		{
			junit.textui.TestRunner.run( suite() );
		}
    }
    
    /**
	 * Common test setup.
	 *
	 * @throws Exception
	 */
    protected void setUp() throws Exception
	{
		super.setUp ();
    }
    
    /**
	 * Tears down the fixture, for example, close a network connection.
	 * This method is called after a test is executed.
	 *
	 * @throws Exception
	 */
    protected void tearDown() throws Exception
	{
    }
    
	/**
	 * <ul>
	 * <li>Create a Resource.
	 * <li>Check it exists.
	 * <li>Delete (unbind) the resx.
	 * <li>Check it has been deleted.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoUnbind() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		testResource.doUnbind();
		assertFalse( resourceExistsOnServer(resourcePath) );
	}
	
	/**
	 * <ul>
	 * <li>Try to unbind a non existing resource.
	 * <li>Exception is thrown?
	 *
	 * @throws Exception
	 */
	//TODO: to be checked with FGA
	public void testDoUnbindOnNonExistingResx() throws Exception
	{
		// create a resource
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		testResource.doUnbind();
		assertFalse( resourceExistsOnServer(resourcePath) );
		
		//unbind a non existing resource.
		try
		{
			testResource.doUnbind();
		}
		catch (WvcmException e)
		{
			assertEquals("Could not doUnbind", ReasonCode.CANNOT_UNBIND_RESOURCE, e.getReasonCode());
			
		}
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Bound this resource to another one.
	 * <li>Delete one of them.
	 * <li>Check the not deleted resource is still there.
	 *
	 * @throws Exception
	 */
	public void testDoUnbindOnBoundResource() throws Exception
	{
		
		// create a resource
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		// create another resource
		location = provider.location( destinationPath );
		Resource testResourceBind = createControllableResource(location);
		assertTrue( resourceExistsOnServer(destinationPath) );
		
		// doBind
		testResource.doBind(location, true);
		
		//doUnbind testResource2
		testResourceBind.doUnbind();
		
		assertTrue( "Resource should exist", resourceExistsOnServer(resourcePath) );
		assertFalse( "Deleted resource still exists", resourceExistsOnServer(destinationPath) );
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource and delete it afterwards.
	 * <li>Try to rebind the deleted resource.
	 * <li>An exception should be thrown.
	 *
	 * @throws Exception
	 */
	public void testDoRebindInvalid() throws Exception
	{
		// create a resource
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		//delete the resource
		testResource.doUnbind();
		
		// create another resource
		// TODO: remove after check with SASIKKA
		// create new loaction ( in order to rebind to a new location)
		//		location = provider.location( destinationPath );
		//		resource = createControllableResource(location);
		//		assertTrue( resourceExistsOnServer(destinationPath) );
		
		//Try to rebind the deleted resource
		try
		{
			testResource.doRebind(location, true);
		}
		catch (WvcmException expected)
		{
			//	fail("Expected Exception: " + expected.toString());
			assertEquals("Could not doRebind", ReasonCode.FORBIDDEN, expected.getReasonCode());
			
		}
	}
	
	/**
	 * <ul>
	 * <li>Create two resources.
	 * <li>Rebind one of them to overwrite the other. Overwrite should be false.
	 * <li>Exception should be thrown.
	 * <li>Repeat Rebind with overwrite true.
	 * <li>Check that the resource has been overwritten and original one is unbound.
	 *
	 * @throws Exception
	 */
	public void testDoRebindOverwriteFalse() throws Exception
	{
		
		resourcePath = rootFolderPath() + "/" + testResx1;
		destinationPath = rootFolderPath() + "/" + testResxCopy;
		
		// create a resource
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		// create another resource
		location = provider.location( destinationPath );
		Resource testResource2 = createControllableResource(location);
		assertTrue( resourceExistsOnServer(destinationPath) );
		try
		{
			testResource.doRebind(location, false);
		}
		catch (WvcmException expected)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + expected.getMessage());
			assertEquals("Could not doRebind",
						 ReasonCode.CONFLICT,
						 expected.getReasonCode());
		}
		
		testResource.doRebind(location, true);
		
		assertFalse( resourceExistsOnServer(resourcePath) );
	}
	
	/**
	 * <ul>
	 * <li>Create a Resource
	 * <li>Read the properties on the resource to get the CONTENT-IDENTIFIER.
	 * <li>Write on that resource (usind the identifier).
	 * <li>Content should have been written.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoWriteContent() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		InputStream testInputStream = null;
		try
		{
			String cid = propsHelper.getContentIdentifier(testResource);
			testInputStream = getClass().getResourceAsStream(testResourcesRoot + "test.xml");
			assertNotNull("INPUT STREAM FOR TEST SHOULD NOT BE NULL", testInputStream);
			
			testResource.doWriteContent(testInputStream, cid );
		}
		finally
		{
			if (testInputStream != null)
				testInputStream.close();
		}
		
		// TODO: ADD CHECK ON THE CONTENT OF THE RESOURCE.
	}
	
	
    
	/**
	 * <ul>
	 * <li>Create a Resource.
	 * <li>Get its content identi
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoWriteContentWrongContentIdentifier() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		long initContentLength = propsHelper.getContentLength(testResource);
		
		// should not update
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test2.xml"),
			cid + "ExtraWrongIdentifier" );
		
		long laterContentLength = propsHelper.getContentLength(testResource);
		assertEquals("Unexpected content length after doWrite with wrong identifier",
					 initContentLength,
					 laterContentLength);
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Delete the resource.
	 * <li>Try to write some content using the resource instance.
	 * <li>Exception should be thrown.
	 *
	 * @throws Exception
	 */
	public void testDoWriteContentOnDeletedResource() throws Exception
	{
		
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		String cid = propsHelper.getContentIdentifier(testResource);
		InputStream testInputStream = null;
		
		//Delete the resource.
		testResource.doUnbind();
		
		try
		{
			
			testInputStream = getClass().getResourceAsStream(testResourcesRoot + "test.xml");
			assertNotNull("INPUT STREAM FOR TEST SHOULD NOT BE NULL", testInputStream);
			
			testResource.doWriteContent(testInputStream, cid );
			fail("Writing on a deleted resource should have thrown an exception.");
		}
		catch (WvcmException expected)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception Message:\n" + expected.getMessage());
			assertEquals("Unexpected exception after writing to a non existing resx.",
						 ReasonCode.WRITE_FAILED,
						 expected.getReasonCode());
		}
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Try to write content to it using a null stream.
	 * <li>Exception should have been thrown.
	 *
	 * @throws Exception
	 */
	public void testDoWriteContentNullStream() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		//InputStream testInputStream = null;
		
		InputStream testInputStream = getClass().getResourceAsStream(
			testResourcesRoot
				+ "nonExistingResx.xml");
		assertNull("INPUT STREAM FOR THIS TEST SHOULD BE NULL", testInputStream);
		
		try
		{
			String cid = propsHelper.getContentIdentifier(testResource);
			
			testResource.doWriteContent(testInputStream, cid );
		}
		catch (WvcmException expected)
		{
			System.out.println(ReasonCodeChecker.getReasonCodeAsString(expected));
			fail("Expected Exception: " + expected.toString());
		}
	}
	
	
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set its content type and content character set.
	 * <li>Write some content in the resource.
	 * <li>Read the resource (including props content type and content char set).
	 * <li>Check those props.
	 * <li>Change those props and write content again.
	 * <li>Read the resorce and check content type.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoWriteContentWithContentType() throws Exception
	{
		String testContentType = "text/plain";
		String testContentCharacterSet = "UTF-8";
		// Create resource for test.
		String testFilePath = rootFolderPath() + "/" + testHelloResx;
		location = provider.location(testFilePath);
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.setContentType( testContentType );
		testResource.setContentCharacterSet( testContentCharacterSet );
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testHelloResx),
			cid );
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.CONTENT_IDENTIFIER,
					PropertyName.CONTENT_TYPE,
					PropertyName.CONTENT_CHARACTER_SET
			}
		);
		
		OutputStream out = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, out );
		
		assertEquals("Unexpected ContentType after read content.",
					 testContentType,
					 testResource.getContentType());
		assertEquals("Unexpected ContentCharacterSet after read content.",
					 testContentCharacterSet,
					 testResource.getContentCharacterSet());
		
		// Change content type and character set.
		testContentType = "text/xml";
		testContentCharacterSet = null;
		testResource.setContentType(testContentType);
		testResource.setContentCharacterSet(testContentCharacterSet);
		
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			testResource.getContentIdentifier() );
		
		out = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, out );
		
		assertEquals("Unexpected ContentType after second read content.",
					 testContentType,
					 testResource.getContentType());
		// Content CharacterSet is not checked since it will not be null as set
		// but it will contain a value set by the server.
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Read resources properties with a null PropertyList.
	 * <li>Check returned property list.
	 *
	 * @throws   Exception
	 *
	 */
	// TODO: CHECK WHAT SHOULD COME IN THIS CASE.
	public void testDoReadPropertiesWithNullList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		PropertyNameList wantedprops = null;
		Resource r = testResource.doReadProperties(wantedprops);
		PropertyNameList returnedProps = r.getPropertyNameList(false);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println(propNames[i].getString());
		}
		// TODO: ADD CHECK & REMOVE SYSTEM.OUT.PRINTLN
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Create a PropertyNameList with some valid properties.
	 * <li>Read the properties.
	 * <li>Check props requested have a value.
	 * <li>Check props not requested does not have a value.
	 *
	 * @throws Exception
	 */
	public void testDoReadPropertiesValidProps() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.CONTENT_IDENTIFIER,
					PropertyName.CONTENT_TYPE,
					PropertyName.CONTENT_CHARACTER_SET
			}
		);
		
		
		Resource r = testResource.doReadProperties(wantedprops);
		
		
		PropertyNameList returnedProps = r.getPropertyNameList(false);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testDoReadPropertiesValidProps");
			System.out.println(propNames[i].getString());
		}
		
		
		
	}
	
	/**
	 * <ul>
	 * <li>Create a Resource.
	 * <li>Create a PropertyNameList with props that should not be contained in
	 * the resource.
	 * <li>Read those properties.
	 * <li>Check values received.
	 *
	 * @throws Exception
	 */
	// TODO: CHECK WHICH PROPERTIES ARE NOT VALID FOR A RESOURCE (VERSION-HISTORY?...)
	
	public void testDoReadInvalidProperties() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.CHECKIN_FORK ,
					PropertyName.VERSION_HISTORY
					
					
			}
		);
		
		
		Resource r = testResource.doReadProperties(wantedprops);
		
		
		PropertyNameList returnedProps = r.getPropertyNameList(false);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testDoReadInvalidProperties");
			System.out.println(propNames[i].getString());
		}
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Without updating any property, get the updated prop list.
	 * <li>List should be empty.
	 *
	 * @throws Exception
	 */
	public void testGetUpdatedProperyListEmpty() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList updatedProps = testResource.getUpdatedPropertyList();
		PropertyName[] pnames = updatedProps.getPropertyNames();
		
		assertEquals("Unexpected number of properties in the updated properties list",
					 0,
					 pnames.length);
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Read some props of the resource.
	 * <li>Update some of them.
	 * <li>Get the updated list.
	 * <li>It should contain the props that have been updated.
	 * <li>Persist property changes.
	 * <li>Get updated property list again. It should be empty.
	 *
	 * @throws Exception
	 */
	public void testGetUpdatedPropertyList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		testResource.setComment( "My Comment" );
		testResource.setDisplayName( "My Display Name" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att, "My Attribute Value" );
		PropertyNameList updatedProps = testResource.getUpdatedPropertyList();
		PropertyName[] pnames = updatedProps.getPropertyNames();
		assertEquals("Unexpected number of properties in the updated properties list.",
					 4,
					 pnames.length );
		testResource.doWriteProperties();
		
		updatedProps = testResource.getUpdatedPropertyList();
		pnames = updatedProps.getPropertyNames();
		assertEquals("Unexpected number of properties in the updated properties list after persist.",
					 0,
					 pnames.length);
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get property Name List for the created resource only attributes (param=true).
	 * <li>Check properties received.
	 *
	 * @throws Exception
	 */
	
	public void testGetPropertyNameListNoPropsTrue() throws Exception
	{
		// TODO: CHECK WHICH PROPERTIES SHOULD COME IN THIS CASE.
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		testResource.setComment( "My Comment" );
		testResource.setDisplayName( "My Display Name" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att, "My Attribute Value" );
		
		testResource.doWriteProperties();
		
		PropertyNameList attributeOnlyProps = testResource.getPropertyNameList(true);
		PropertyName[] pnames = attributeOnlyProps.getPropertyNames();
		
		
		assertEquals("Unexpected number of properties in the properties list",
					 1,
					 pnames.length);
		
		assertEquals("Expected to get only the Attributes", pnames[0].getString(), "myAttribute");
		
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get property Name List for the created resource with attributes (param=false).
	 * <li>Check properties received.
	 *
	 * @throws Exception
	 */
	
	public void testGetPropertyNameListNoPropsFalse() throws Exception
	{
		// TODO: CHECK WHICH PROPERTIES SHOULD COME IN THIS CASE.
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		testResource.setComment( "My Comment" );
		testResource.setDisplayName( "My Display Name" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att, "My Attribute Value" );
		
		testResource.doWriteProperties();
		
		PropertyNameList attributeOnlyProps = testResource.getPropertyNameList(false);
		PropertyName[] pnames = attributeOnlyProps.getPropertyNames();
		
		
		assertEquals("Unexpected number of properties in the properties list",
					 4,
					 pnames.length);
		/**
		 for(int i=0; i < pnames.length; i++)
		 {
		 System.out.println("testGetPropertyNameListNoPropsFalse");
		 System.out.println(pnames[i].getString());
		 }
		 **/
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Create a PropertyNameList with ALL_ATTRIBUTES
	 * <li>Read properties of resource with the created list.
	 * <li>GetPropertyNameList only Attributes.
	 * <li>Check props received.
	 *
	 * @throws Exception
	 */
	
	public void testGetProperyNameListTrue() throws Exception
	{
		
		// Create resource for test.
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.ALL_ATTRIBUTES,
			}
		);
		
		Resource r = testResource.doReadProperties(wantedprops);
		
		//Only attributes are retrieved
		PropertyNameList returnedProps = r.getPropertyNameList(true);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testGetProperyNameListTrue");
			System.out.println(propNames[i].getString());
		}
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Create a PropertyNameList with ALL_ATTRIBUTES
	 * <li>Read properties of resource with the created list.
	 * <li>GetPropertyNameList not only Attributes.
	 * <li>Check props received.
	 *
	 * @throws Exception
	 */
	
	public void testGetProperyNameListFalse() throws Exception
	{
		// Create resource for test.
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.ALL_ATTRIBUTES,
			}
		);
		
		Resource r = testResource.doReadProperties(wantedprops);
		
		PropertyNameList returnedProps = r.getPropertyNameList(false);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testGetProperyNameListFalse");
			System.out.println(propNames[i].getString());
		}
		
		
		
		
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Create a PropertyNameList with some resources.
	 * <li>Read properties of resource with the created list.
	 * <li>Get PropertyNameList (once with only Attributes and once more with all).
	 * <li>Check properties received.
	 *
	 * @throws Exception
	 */
	
	public void testGetPropertyNameListSomeProps() throws Exception
	{
		// Create resource for test.
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops =
			new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.CONTENT_IDENTIFIER,
					PropertyName.CONTENT_TYPE,
					PropertyName.CONTENT_CHARACTER_SET
			}
		);
		
		Resource r = testResource.doReadProperties(wantedprops);
		
		PropertyNameList returnedProps = r.getPropertyNameList(false);
		PropertyName[] propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testGetPropertyNameListSomeProps-false");
			System.out.println(propNames[i].getString());
		}
		//Only attributes are retrieved
		returnedProps = r.getPropertyNameList(true);
		propNames = returnedProps.getPropertyNames();
		
		for(int i=0; i < propNames.length; i++)
		{
			System.out.println("testGetPropertyNameListSomeProps-true");
			System.out.println(propNames[i].getString());
		}
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a Resource.
	 * <li>Write some content in it.
	 * <li>Read content and compare with the written one.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoReadContent() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		String testFileName = testResourcesRoot + "test.xml";
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testFileName),
			cid );
		
		String contentForTest = null;
		contentForTest = fileHelper.getResourceAsString(testFileName);
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(null, content);
		
		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.areXmlStringEqual(contentForTest,
											   content.toString()));
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Delete it.
	 * <li>Try to write some content on it.
	 * <li>Exception should be thrown.
	 *
	 * @throws Exception
	 */
	public void testDoReadContentOnDeleted() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		String cid = propsHelper.getContentIdentifier(testResource);
		
		testResource.doUnbind();
		
		String testFileName = testResourcesRoot + "test.xml";
		
		try
		{
			testResource.doWriteContent(
				getClass().getResourceAsStream(testFileName),
				cid );
		}
		catch (WvcmException expected)
		{
			
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception Message:\n" + expected.getMessage());
			
			assertEquals("Expected exception",
						 ReasonCode.WRITE_FAILED,
						 expected.getReasonCode());
		}
		
		
	}
	
	
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes on the resource and persist the changes.
	 * <li>Get the propertynamelist.
	 * <li>Check properties received.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testSetAttributesValid() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource.setAttribute( att1, "Hello" );
		testResource.setAttribute( att2, "World" );
		
		// write properties is used to persist the changes in the server.
		testResource.doWriteProperties();
		
		PropertyNameList allAttributes;
		int n;
		allAttributes = testResource.getPropertyNameList(true);
		n = allAttributes.getPropertyNames().length;
		assertEquals("Unexpected attribute list read from resource.",
					 2,
					 n);
		for( int i=0; i < n; i++)
		{
			if ((allAttributes.getPropertyNames()[i] != att1) &&
					(allAttributes.getPropertyNames()[i] != att2))
			{
				fail("Unexpected Attribute Received.");
			}
		}
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes and some props on the resource and
	 * persist the changes.
	 * <li>Get PropertyNamesList only with the attributes.
	 * <li>Check list returned only contains the attributes.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testGetPropertyNameListOnlyAtts() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource .setComment( "My Comment" );
		testResource .setDisplayName( "My Display Name" );
		testResource .setContentLanguage( new Locale("de","") );
		testResource .setAttribute( att1, "Hello" );
		testResource .setAttribute( att2, "World" );
		testResource .doWriteProperties();
		
		PropertyNameList attnames = testResource .getPropertyNameList(true);
		int n = attnames.getPropertyNames().length;
		assertTrue( n == 2 );
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes and some props on the resource and
	 * persist the changes.
	 * <li>Get PropertyNamesList not only with the attributes.
	 * <li>Check list returned contains the attributes and the properties.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testGetPropertyNameListPropsAndAtts() throws Exception
	{
		location = provider.location( resourcePath );
		testResource  = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource .setComment( "My Comment" );
		testResource .setDisplayName( "My Display Name" );
		testResource .setContentLanguage( new Locale("de","") );
		testResource .setAttribute( att1, "Hello" );
		testResource .setAttribute( att2, "World" );
		testResource .doWriteProperties();
		
		PropertyNameList propnames = testResource .getPropertyNameList(false);
		int n = propnames.getPropertyNames().length;
		assertTrue( n == 5 );
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes on the resource and persist the changes.
	 * <li>Check attributes are there.
	 * <li>Remove one of the properties.
	 * <li>Read attributes again and check it is deleted.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testRemoveAttributes() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource.setAttribute( att1, "Hello" );
		testResource.setAttribute( att2, "World" );
		testResource.doWriteProperties();
		
		PropertyNameList attnames = testResource.getPropertyNameList(true);
		int n = attnames.getPropertyNames().length;
		assertTrue( n == 2 );
		
		testResource.removeAttribute( att1 );
		testResource.doWriteProperties();
		
		attnames = testResource.getPropertyNameList(true);
		n = attnames.getPropertyNames().length;
		assertTrue( n == 1 );
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Create an attribute and remove it from the resource.
	 * <li>Exception should be thrown.
	 *
	 * @throws Exception
	 */
	
	public void testRemoveNonAddedAttribute() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		
		testResource.setAttribute( att1, "Hello" );
		try
		{
			testResource.removeAttribute( att1 );
		}
		catch (Exception expected)
		{
			
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception Message:\n" + expected.getMessage());
			
			
			System.out.println("testRemoveNonAddedAttribute");
			System.out.println(expected.toString());
			
		}
		
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Remove attribute with null as parameter.
	 * <li>Exception should be thrown.
	 *
	 * @throws Exception
	 */
	
	public void testRemoveNullAttribute() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		try
		{
			testResource.removeAttribute( null );
		}
		catch (Exception expected)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception Message:\n" + expected.getMessage());
			
			System.out.println("testRemoveNullAttribute");
			System.out.println(expected.toString());
			
		}
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Write some content in it.
	 * <li>Read properties specifying CREATOR_DISPLAY_NAME.
	 * <li>GetCreatorDisplayName should fail since resource has not been put under
	 * version control
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testGetCreatorDisplayNameFails() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"), cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.CREATOR_DISPLAY_NAME} );
		testResource = testResource.doReadProperties(wantedprops);
		try
		{
			testResource.getCreatorDisplayName();
			fail("Exception not thrown regarding not available creator-display-name.");
		}
		catch (WvcmException we)
		{
			assertEquals("Unexpected Reason Code in exception.",
						 ReasonCode.VALUE_UNAVAILABLE,
						 we.getReasonCode());
		}
	}
	
	
	/**
	 * <ul>
	 * <li>Create a Resource and write some content in it.
	 * <li>Read Property CREATION_DATE and check that it is OK.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testGetCreationDate() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.CREATION_DATE} );
		
		testResource = testResource.doReadProperties(wantedprops);
		Date now = new Date();
		Date resxCreationDate = testResource.getCreationDate();
		assertEquals("Unexpected creation date.",
					 now.getDay(),
					 resxCreationDate.getDay());
	}
	
	
	/**
	 * <ul>
	 * <li>Create a Resource and write some content in it.
	 * <li>Read Property CREATION_DATE and LAST_MODIFIED.
	 * <li>Check that lastModified is the same as creation date.
	 *
	 * @throws   Exception
	 *
	 */
	public void testGetLastModifiedNoModification() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.LAST_MODIFIED, PropertyName.CREATION_DATE} );
		
		testResource = testResource.doReadProperties(wantedprops);
		assertTrue("Unexpected Last Modified.",
				   testResource.getLastModified().equals(testResource.getCreationDate()));
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Modify it (so that its last modified property should be changed).
	 * <li>Check last modified.
	 * <li>Modify it again(so that its last modified property should be changed).
	 * <li>Check last modified again.
	 *
	 * @throws Exception
	 */
	public void testGetLastModified() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		Thread.sleep(1000);
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.LAST_MODIFIED,
					PropertyName.CREATION_DATE} );
		
		testResource = testResource.doReadProperties(wantedprops);
		Date firstLastModified = testResource.getLastModified();
		try
		{
			assertTrue("Unexpected Last Modified.",
					   firstLastModified.after(testResource.getCreationDate()));
		}
		catch (AssertionFailedError afe)
		{
			if (logger.isLoggable(Level.CONFIG))
			{
				logger.config("\nResource Last Modified:\t" + testResource.getLastModified()
								  + "Resource Creation Date:\t" + testResource.getCreationDate());
			}
			throw afe;
		}
		
		// Modify again the content of the file.
		Thread.sleep(1000);
		cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		testResource = testResource.doReadProperties(wantedprops);
		
		try
		{
			assertTrue("Unexpected Last Modified.",
					   testResource.getLastModified().after(firstLastModified));
		}
		catch (AssertionFailedError afe)
		{
			if (logger.isLoggable(Level.CONFIG))
			{
				logger.config("\nResource First Last Modified:\t" + firstLastModified
								  + "Resource Last Modified:\t" + testResource.getCreationDate());
			}
			throw afe;
		}
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Get property ContentLanguage.
	 * <li>Check that the property returned is not null or an empty string.
	 * <li>Set another content language in the resource.
	 * <li>Get againg property ContentLanguage.
	 * <li>Check that the property returned is equal to the set one.
	 *
	 * @throws   Exception
	 *
	 */
	public void testSetGetContentLanguage() throws Exception
	{
		location = provider.location( resourcePath );
		testResource= createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList( new PropertyName[]
															{
					PropertyName.CONTENT_LANGUAGE} );
		
		testResource = testResource.doReadProperties(wantedprops);
		Locale contentLanguage = testResource.getContentLanguage();
		assertTrue("Content language not received as expected.",
					   (contentLanguage != null) && (contentLanguage.toString().length()>0));
		
		Locale testContentLanguage = Locale.ITALIAN;
		testResource.setContentLanguage(testContentLanguage);
		testResource.doWriteProperties();
		testResource = testResource.doReadProperties(wantedprops);
		assertEquals("Content language not received as expected.",
					 testContentLanguage,
					 testResource.getContentLanguage());
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Get property Content Character Set.
	 * <li>Check that the property returned is not null or an empty string.
	 *
	 * @throws   Exception
	 *
	 */
	// TODO: CHECK WHY IT RETURNS A VALUE ONLY AFTER A READ CONTENT...
	// TODO: ADD SET CONTENT CHARACTER SET PART. REMOVE PART FROM OTHER TEST CASE.
	public void testSetGetContentCharacterSet() throws Exception
	{
		location = provider.location( resourcePath );
		testResource= createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.CONTENT_CHARACTER_SET} );
		
		//testResource = testResource.doReadProperties(wantedprops);
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(wantedprops, content);
		
		String contentCharSet = testResource.getContentCharacterSet();
		assertTrue("Unexpected character set.",
				   contentCharSet != null && contentCharSet.length() > 0);
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Write some properties in the resource.
	 * <li>Check updated property list contains those props.
	 * <li>DoWriteProperties (makes properties persistent in server).
	 * <li>Check updated property list is empty.
	 * <li>Check that the properties set is correct.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoWriteProperties() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		testResource.setComment( "My Comment" );
		testResource.setDisplayName( "My Display Name" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att, "My Attribute Value" );
		
		testResource.doWriteProperties();
		
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.DISPLAY_NAME,
						PropertyName.CONTENT_LANGUAGE,
						att} );
		testResource = testResource.doReadProperties(wantedprops);
		assertEquals( "My Comment", testResource.getComment() );
		assertEquals( "My Display Name", testResource.getDisplayName() );
		assertEquals( "de", testResource.getContentLanguage().toString() );
		assertEquals( "My Attribute Value", testResource.getAttribute(att) );
	}
	
	
	/**
	 * <ul>
	 * <li>Create a test resource.
	 * <li>Remove the destination resource (if existing).
	 * <li>Set some properties and attributes in resource.
	 * <li>Copy resource to destination with overwrite true.
	 * <li>Check properties set. Properties are set to default values.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoCopyOverwriteNothingToOverwrite() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		// remove the destination resource in case it exists.
		Resource copyDestinationResx = provider.location(destinationPath).controllableResource();
		copyDestinationResx.doUnbind();
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource.setComment( "My Comment" );
		testResource.setContentLanguage( new Locale("en","") );
		testResource.setAttribute( att1, "Hello" );
		testResource.setAttribute( att2, "World" );
		testResource.doWriteProperties();
		
		testResource.doCopy(destinationPath, true);
		
		PropertyNameList wantedprops =
			new PropertyNameList(new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.CONTENT_LANGUAGE,
						att1,
						att2
				}
								);
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, content );
		String contentAsString = content.toString();
		content = new ByteArrayOutputStream();
		copyDestinationResx = copyDestinationResx.doReadContent( wantedprops, content );
		String destContentAsString = content.toString();
		
		assertEquals("Unexpected attribute 1 after copy.",
					 testResource.getAttribute(att1),
					 copyDestinationResx.getAttribute(att1));
		assertEquals("Unexpected attribute 2 after copy.",
					 testResource.getAttribute(att2),
					 copyDestinationResx.getAttribute(att2));
		assertEquals("Unexpected comment after copy.",
					 "",
					 copyDestinationResx.getComment());
		assertEquals("Unexpected content language after copy.",
					 testResource.getContentLanguage(),copyDestinationResx.getContentLanguage());
		assertEquals("Unexpected content after copy.",
					 contentAsString, destContentAsString);
		
		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.areXmlStringEqual(contentAsString,
											   destContentAsString));
	}
	
	
	/**
	 * <ul>
	 * <li>Create a test resource.
	 * <li>Remove the destination resource (if existing).
	 * <li>Set some properties and attributes in resource.
	 * <li>Copy resource to destination with overwrite false (there is nothing
	 * to overwrite).
	 * <li>Check properties set.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testDoCopyNoOverwriteNothingToOverwrite() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		// remove the destination resource in case it exists.
		Resource copyDestinationResx = provider.location(destinationPath).controllableResource();
		copyDestinationResx.doUnbind();
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"), cid );
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource.setComment( "My Comment" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att1, "Hello" );
		testResource.setAttribute( att2, "World" );
		testResource.doWriteProperties();
		
		//overwrite set to false although there is nothing to overwrite.
		testResource.doCopy(destinationPath, false);
		
		PropertyNameList wantedprops =
			new PropertyNameList(new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.CONTENT_LANGUAGE,
						att1,
						att2
				}
								);
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, content );
		String contentAsString = content.toString();
		content = new ByteArrayOutputStream();
		
		copyDestinationResx = copyDestinationResx.doReadContent( wantedprops, content );
		String destContentAsString = content.toString();
		
		assertEquals(testResource.getAttribute(att1),copyDestinationResx.getAttribute(att1));
		assertEquals(testResource.getAttribute(att2),copyDestinationResx.getAttribute(att2));
		assertEquals("Unexpected comment found in copied resource.",
					 "",
					 copyDestinationResx.getComment());
		assertEquals(testResource.getContentLanguage(),copyDestinationResx.getContentLanguage());
		
		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.areXmlStringEqual(contentAsString,
											   destContentAsString));
	}
	
	
	/**
	 * <ul>
	 * <li>Create a two resources.
	 * <li>Copy one with overwrite true, and overwrite the second resource.
	 * <li>Check that the resource has been copied.
	 *
	 * @throws Exception
	 */
	
	public void testDoCopyOverwriteSomething() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		Location	destinationLocation = provider.location( destinationPath );
		Resource testResource2 = createControllableResource(destinationLocation);
		assertTrue( resourceExistsOnServer(destinationPath) );
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"), cid );
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		testResource.setComment( "My Comment" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att1, "Hello" );
		testResource.setAttribute( att2, "World" );
		
		testResource.doWriteProperties();
		try
		{
			//overwrite
			testResource.doCopy(destinationPath, true);
		}
		catch (WvcmException we)
		{
			System.out.println(ReasonCodeChecker.getReasonCodeAsString(we));
			fail("Exception : " + we.toString());
		}
		
		PropertyNameList wantedProps =
			new PropertyNameList(new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.CONTENT_LANGUAGE,
						att1,
						att2});
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedProps, content );
		String contentAsString = content.toString();
		content = new ByteArrayOutputStream();
		
		testResource2 = testResource2.doReadContent( wantedProps, content );
		String destContentAsString = content.toString();
		
		assertEquals(testResource.getAttribute(att1),testResource2.getAttribute(att1));
		assertEquals(testResource.getAttribute(att2),testResource2.getAttribute(att2));
		// FIXME
		assertEquals("Unexpected comment found in copied resource.",
					 "",
					 testResource2.getComment());
		assertEquals(testResource.getContentLanguage(),testResource2.getContentLanguage());
		
		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.areXmlStringEqual(contentAsString,
											   destContentAsString));
		
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create two resources.
	 * <li>Copy one of them trying to overwrite the second, but with overwrite false.
	 * <li>Check that copy was not successful.
	 *
	 * @throws Exception
	 */
	
	public void testDoCopyOverwriteFalseOverSomething() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		Location destinationLocation = provider.location( destinationPath );
		Resource testResource2 = createControllableResource(destinationLocation);
		assertTrue( resourceExistsOnServer(destinationPath) );
		
		try
		{
			//do not overwrite
			testResource.doCopy(destinationPath, false);
		}
		catch (WvcmException expected)
		{
			
			assertEquals("Could not copy", ReasonCode.CONFLICT, expected.getReasonCode());
		}
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create one resource.
	 * <li>Delete it.
	 * <li>Try a DoCopy with the resource instance.
	 * <li>Check Exception received.
	 *
	 * @throws Exception
	 */
	
	public void testDoCopyDeletedSource() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		testResource.doUnbind();
		assertFalse( resourceExistsOnServer(resourcePath) );
		
		try
		{
			testResource.doCopy(resourcePath, true);
		}
		catch (WvcmException expected)
		{
			assertEquals("Could not copy", ReasonCode.FORBIDDEN, expected.getReasonCode());
		}
	}
	
	
	/**
	 * <ul>
	 * <li>Create two resources.
	 * <li>Put one of them under version control
	 * <li>Try to copy the first resource with overwrite true over the resource
	 * under version control.
	 * <li>Exception should be thrown.
	 * <li>Repeat the same but with overwrite false.
	 *
	 * @throws Exception
	 */
	//TODO: to be checked by FGA
	
	public void testDoCopyOverCheckedResource() throws Exception
	{
		
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		//	Location destinationLocation = provider.location( destinationPath );
		//	Resource testResource2 = createControllableResource(destinationLocation);
		//  can I put any resource under vesrion control or only ontrollableResource
		
		location = provider.location( destinationPath );
		resource = createControllableResource(location);
		resource.doControl();
		assertTrue( resourceExistsOnServer(destinationPath) );
		
		try
		{
			testResource.doCopy(destinationPath, false);
		}
		catch (WvcmException expected)
		{
			assertEquals("Could not copy when overwrite=false", ReasonCode.CONFLICT, expected.getReasonCode());
		}
		
		try
		{
			testResource.doCopy(destinationPath, true);
		}
		catch (WvcmException we)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Try to copy the resource over itself with overwrite true.
	 * <li>Exception should be thrown.
	 * <li>Same with overwrite false.
	 *
	 * @throws Exception
	 */
	public void testDoCopyOverItselfOverwriteTrue() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		assertTrue( resourceExistsOnServer(resourcePath) );
		
		try
		{
			testResource.doCopy(resourcePath, true);
		}
		catch (WvcmException expected)
		{
			assertEquals("Could not copy", ReasonCode.FORBIDDEN, expected.getReasonCode());
		}
	}
	
	
	/**
	 * <ul>
	 * <li>Create two folders and a resource in one of them.
	 * <li>Bind the resource to the other folder (without overwrite).
	 * <li>Check that the resource is present in both locations.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoBind() throws Exception
	{
		String originFolderName = "originFolder";
		String destinationFolderName = "destinationFolder";
		Iterator members;
		
		// Create origin and destination folders.
		location = provider.location( rootFolderPath() + "/" + originFolderName );
		ControllableFolder originFolder = createControllableFolder(location);
		location = provider.location( rootFolderPath() + "/" + destinationFolderName );
		ControllableFolder destinationFolder = createControllableFolder(location);
		
		// Create resource in origin folder.
		location = originFolder.location().child(testResx1);
		testResource = createControllableResource(location);
		
		Location destinationLoc = destinationFolder.location().child(testResx1);
		
		// bind resource to destination folder.
		testResource.doBind(destinationLoc, false);
		
		// Check that file is still in the origin folder.
		members = originFolder.doReadMemberList( null, false );
		Resource tempResx;
		boolean resourceExists = false;
		while( members.hasNext() )
		{
			tempResx = (Resource)members.next();
			if (tempResx.location().string().endsWith(originFolderName + "/" + testResx1))
				resourceExists = true;
		}
		assertTrue("Origin resource has not been found after a doBind.", resourceExists);
		
		resourceExists = false;
		members = destinationFolder.doReadMemberList( null, false );
		
		while( members.hasNext() )
		{
			tempResx = (Resource)members.next();
			if (tempResx.location().string().endsWith(destinationFolderName + "/" + testResx1))
				resourceExists = true;
		}
		assertTrue("Origin resource has not been found after a doBind.", resourceExists);
	}
	
	/**
	 * <ul>
	 * <li>Create two folders and one resource.
	 * <li>Set some property/attribute on the resource.
	 * <li>Bind the resource to the second folder.
	 * <li>Check that properties in the bound resource are the same as in the
	 * original one.
	 *
	 * @throws Exception
	 */
	public void testDoBindCheckProperties() throws Exception
	{
		String originFolderName = "originFolder";
		String destinationFolderName = "destinationFolder";
		
		
		// Create origin and destination folders.
		location = provider.location( rootFolderPath() + "/" + originFolderName );
		ControllableFolder originFolder = createControllableFolder(location);
		location = provider.location( rootFolderPath() + "/" + destinationFolderName );
		ControllableFolder destinationFolder = createControllableFolder(location);
		
		// Create resource in origin folder.
		location = originFolder.location().child(testResx1);
		testResource = createControllableResource(location);
		
		// prepare location to bind the test resource.
		Location destinationLoc = destinationFolder.location().child(testResx1);
		
		
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		String testComment = "My Comment";
		testResource .setComment( testComment );
		String testDisplayName = "My Display Name";
		testResource .setDisplayName( testDisplayName );
		Locale testLocale = new Locale("de","");
		testResource .setContentLanguage( testLocale );
		String testAtt1Value = "Hello";
		testResource .setAttribute( att1, testAtt1Value );
		String testAtt2Value = "World";
		testResource .setAttribute( att2, testAtt2Value );
		testResource .doWriteProperties();
		
		
		// bind resource to destination folder.
		testResource.doBind(destinationLoc, false);
		
		//Check that properties in the bound resource are the same as in the
		//original one.
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.DISPLAY_NAME,
						PropertyName.CONTENT_LANGUAGE,
						att1,
						att2} );
		Resource boundResource = destinationLoc.resource();
		boundResource = boundResource.doReadProperties(wantedprops);
		
		assertEquals("Unexpected comment in bound resource.",
					 testComment,
					 boundResource.getComment());
		assertEquals("Unexpected display name in bound resource.",
					 testDisplayName,
					 boundResource.getDisplayName());
		assertEquals("Unexpected content language in bound resource.",
					 testLocale,
					 boundResource.getContentLanguage());
		assertEquals("Unexpected attribute 1 value in bound resource.",
					 testAtt1Value,
					 boundResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in bound resource.",
					 testAtt2Value,
					 boundResource.getAttribute(att2));
	}
	
	/**
	 * <ul>
	 * <li>Create two folders and two resources.
	 * <li>Try to bind one of the resources to the other with overwrite false.
	 * <li>WvcmException should be thrown accordingly.
	 * <li>Try again to bind one of the resources to the other with overwrite true.
	 * <li>Resource should have been bound.
	 * <li>Check that both resources represent the same one.
	 * <li>Change content to one of the resources. Check content in the other.
	 * <li>Content should be the same.
	 * <li>Change properties to one of the resx. Check properties in the other.
	 * <li>Property changed should have the same value in both resx.
	 *
	 * @throws Exception
	 */
	public void testDoBindOverwrite() throws Exception
	{
		String originFolderName = "originFolder";
		String destinationFolderName = "destinationFolder";
		
		
		// Create origin and destination folders.
		location = provider.location( rootFolderPath() + "/" + originFolderName );
		ControllableFolder originFolder = createControllableFolder(location);
		location = provider.location( rootFolderPath() + "/" + destinationFolderName );
		ControllableFolder destinationFolder = createControllableFolder(location);
		
		// Create resource in origin folder.
		location = originFolder.location().child(testResx1);
		testResource = createControllableResource(location);
		
		//Create resource in destination folder
		location = destinationFolder.location().child(testResxCopy);
		Resource testResource2 = createControllableResource(location);
		
		//Try to bind one of the resources to the other with overwrite false.
		try
		{
			testResource.doBind(location, false);
		}
		catch (WvcmException expected)
		{
			assertEquals("getSupportedFolderList fails", ReasonCode.CANNOT_OVERWRITE, expected.getReasonCode());
		}
		
		
		//Try to bind one of the resources to the other with overwrite true.
		try
		{
			testResource.doBind(location, true);
		}
		catch (WvcmException we)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			
			fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
		
		fail("need to do content checking/property checking");
		
		
		
	}
	
	/**
	 * <ul>
	 * <li>Create one folder and a resource.
	 * <li>Try to bind the resource to multiple destinations (same folder,
	 * different names).
	 * <li>Check that resources have been bound correctly.
	 *
	 * @throws Exception
	 */
	public void testDoBindMultiple() throws Exception
	{
		String originFolderName = "originFolder";
		String destinationFolderName1 = "destinationFolder1";
		String destinationFolderName2 = "destinationFolder2";
		String destinationFolderName3 = "destinationFolder3";
		
		
		// Create origin and destination folders.
		location = provider.location( rootFolderPath() + "/" + originFolderName );
		ControllableFolder originFolder = createControllableFolder(location);
		
		location = provider.location( rootFolderPath() + "/" + destinationFolderName1 );
		ControllableFolder destinationFolder1 = createControllableFolder(location);
		
		location = provider.location( rootFolderPath() + "/" + destinationFolderName2 );
		ControllableFolder destinationFolder2 = createControllableFolder(location);
		
		location = provider.location( rootFolderPath() + "/" + destinationFolderName3);
		ControllableFolder destinationFolder3 = createControllableFolder(location);
		
		
		// Create resource in origin folder.
		location = originFolder.location().child(testResx1);
		testResource = createControllableResource(location);
		
		//why ??
		Location destinationLoc1 = destinationFolder1.location().child(testResx1);
		Location destinationLoc2 = destinationFolder2.location().child(testResx1);
		Location destinationLoc3 = destinationFolder3.location().child(testResx1);
		
		// bind resource to destination folder.
		testResource.doBind(destinationLoc1, false);
		assertTrue( resourceExistsOnServer(rootFolderPath() + "/" + destinationFolderName1 ) );
		
		testResource.doBind(destinationLoc2, false);
		assertTrue( resourceExistsOnServer(rootFolderPath() + "/" + destinationFolderName2 ) );
		
		testResource.doBind(destinationLoc3, false);
		assertTrue( resourceExistsOnServer(rootFolderPath() + "/" + destinationFolderName3 ) );
		
		
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Rebind it to a valid destination (no resource in destination) with
	 * overwrite true.
	 * <li>Check that initial location has been unbound and destination contains
	 * the resource. Properties should remain the same.
	 * Test again with overwrite false (same result).
	 *
	 * @throws Exception
	 */
	public void testDoRebindValid() throws Exception
	{
		location = provider.location(resourcePath);
		testResource  = createControllableResource(location);
		location = provider.location(destinationPath);
		Resource destResource = createControllableResource(location);
		
		String testFileContent = fileHelper.getResourceAsString(testResourcesRoot + testResx1);
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		String comment = "testDoRebindValid Comment";
		String displayName = "testDoRebindValid Display Name";
		Locale testLocale = Locale.getDefault();
		String att1Value = "testDoRebindValid Attribute 1";
		String att2Value = "testDoRebindValid Attribute 2";
		testResource.setComment( comment);
		testResource.setDisplayName(displayName);
		testResource.setContentLanguage(testLocale);
		testResource.setAttribute( att1, att1Value);
		testResource.setAttribute( att2, att2Value);
		testResource.doWriteProperties();
		
		testResource.doRebind(provider.location(destinationPath), true);
		
		assertFalse("Resource has been rebound, but it is still on origin location.",
					resourceExistsOnServer(resourcePath));
		
		// Check that the new location points to the destination.
		assertEquals( destinationPath, testResource.location().string() );
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.DISPLAY_NAME,
						PropertyName.CONTENT_LANGUAGE,
						att1,
						att2} );
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		destResource = destResource.doReadContent( wantedprops, content );
		assertTrue("Content of resource copied first differs from original.",
				   xmlHelper.areXmlStringEqual(testFileContent,
											   content.toString()));
		
		assertEquals("Unexpected comment in rebound resource.",
					 comment,
					 destResource.getComment());
		assertEquals("Unexpected display name in rebound resource.",
					 displayName,
					 destResource.getDisplayName());
		assertEquals("Unexpected content language in rebound resource.",
					 testLocale,
					 destResource.getContentLanguage());
		assertEquals("Unexpected attribute 1 value in rebound resource.",
					 att1Value,
					 destResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in rebound resource.",
					 att2Value,
					 destResource.getAttribute(att2));
		
		// repeat the test with overwrite false (destination is empty)
		testResource.doRebind(provider.location(resourcePath), false);
		
		assertFalse("Resource has been rebound, but it is still on origin location.",
					resourceExistsOnServer(destinationPath));
		
		// Check that the new location points to the destination.
		assertEquals( resourcePath, testResource.location().string() );
		
		content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, content );
		assertEquals("Unexpected comment in rebound resource.",
					 comment,
					 testResource.getComment());
		assertEquals("Unexpected display name in rebound resource.",
					 displayName,
					 testResource.getDisplayName());
		assertEquals("Unexpected content language in rebound resource.",
					 testLocale,
					 testResource.getContentLanguage());
		assertEquals("Unexpected attribute 1 value in rebound resource.",
					 att1Value,
					 testResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in rebound resource.",
					 att2Value,
					 testResource.getAttribute(att2));
		
		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.areXmlStringEqual(testFileContent,
											   content.toString()));
	}
	
	/**
	 * <ul>
	 * <li>Create a test resource.
	 * <li>Write some content in it.
	 * <li>Read resource for property WORKSPACE_FOLDER_LIST.
	 * <li>Try to get workspace folder list.
	 * <li>List returned should not be empty and folder returned should exist.
	 *
	 * @throws Exception
	 */
	// TODO: ADD DOCUMENTATION
	// TODO: REMOVE CONSOLE OUTPUT. ADD ASSERTIONS.
	// TODO: CHANGE ControllableResource with a Resource variable.
	public void testGetWorkspaceFolderList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.WORKSPACE_FOLDER_LIST} );
		
		testResource = testResource.doReadProperties(wantedprops);
		
		try
		{
			List workspaceFolderList = testResource.getWorkspaceFolderList();
			assertNotNull("RETURNED WORKSPACE FOLDER LIST IS UNEXPECTEDLY NULL.",
						  workspaceFolderList);
			// TODO: ADD CHECK IN FOLDERS RETURNED. THEY SHOULD EXIST.
		}
		catch (WvcmException e)
		{
			// TODO: CHECK IF SERVER IMPLEMENTS THIS FEATURE.
			if( e.getReasonCode() == ReasonCode.VALUE_UNAVAILABLE )
				assertTrue(
					"IGNORE FOR NOW! This testcase may fail while the server does not yet properly implement the DAV:workspace-collection-set property",
					false );
			else
				throw e;
		}
	}
	
	/**
	 * <ul>
	 * <li>Create two folders. Put a resource in one of them and add a bind to
	 * it on the second folder.
	 * <li>Get the resource identifiers for both resources.
	 * <li>Check that both identifiers are the same.
	 *
	 * @throws Exception
	 */
	// TODO: ADD DOCUMENTATION
	// TODO: REMOVE CONSOLE OUTPUT. ADD ASSERTIONS.
	// TODO: CHANGE ControllableResource with a Resource variable.
	public void testGetResourceIdentifier() throws Exception
	{
		String testFolderFoo = "foo";
		String testFolderBar = "bar";
		
		location = provider.location( rootFolderPath()
										 + "/" + testFolderFoo );
		Folder fooFolder = createControllableFolder(location);
		
		location = provider.location( rootFolderPath()
										 + "/" + testFolderFoo
										 + "/" + testFolderBar );
		
		ControllableFolder barFolder = (ControllableFolder)location.folder();
		barFolder.doCreateResource();
		location = provider.location( fooFolder.location().string() + "/" + testResx1 );
		testResource = createControllableResource(location);
		
		location = provider.location( barFolder.location().string() + "/test2.xml" );
		Resource testFile2Resource = location.controllableResource();
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		testResource.doBind(barFolder.location().child("test2.xml"), true);
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.RESOURCE_IDENTIFIER} );
		
		// Read properties from resource in fooFolder.
		testResource = testResource.doReadProperties(wantedprops);
		testFile2Resource = testFile2Resource.doReadProperties(wantedprops);
		assertEquals("Unexpected difference in the resource identifiers of two bound resources.",
					 testResource.getResourceIdentifier(),
					 testFile2Resource.getResourceIdentifier());
	}
	
	/**
	 * <ul>
	 * <li>Create two folders. Add a resource to one of them. Bind the resource
	 * to the other folder.
	 * <li>Get the parent binding list of both resources.
	 * <li>Check that both resources have the same bound members in their lists.
	 *
	 * @throws Exception
	 */
	public void testGetParentBindingList() throws Exception
	{
		String testFolderFoo = "foo";
		String testFolderBar = "bar";
		
		location = provider.location( rootFolderPath()
										 + "/" + testFolderFoo );
		ControllableFolder fooFolder = createControllableFolder(location);
		
		location = provider.location( rootFolderPath()
										 + "/" + testFolderFoo
										 + "/" + testFolderBar );
		ControllableFolder barFolder = (ControllableFolder)location.folder();
		barFolder.doCreateResource();
		location = provider.location( fooFolder.location().string() + "/" + testResx1 );
		testResource = createControllableResource(location);
		
		location = provider.location( barFolder.location().string() + "/test2.xml" );
		Resource testFile2Resource = location.controllableResource();
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		testResource.doBind(barFolder.location().child("test2.xml"), true);
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.PARENT_BINDING_LIST} );
		
		// Read properties from resource in fooFolder.
		testResource = testResource.doReadProperties(wantedprops);
		
		// Read properties from resource in barFolder
		testFile2Resource = testFile2Resource.doReadProperties(wantedprops);
		
		// Check that both resources has the same bound members.
		Iterator testResourceBindingListIterator = testResource.getParentBindingList().iterator();
		Iterator testResource2BindingListIterator = testFile2Resource.getParentBindingList().iterator();
		Folder.Binding pb;
		Folder.Binding pb2;
		Resource tempBoundMember1;
		Resource tempBoundMember2;
		while (testResourceBindingListIterator.hasNext() &&
			   testResource2BindingListIterator.hasNext())
		{
			pb = (Folder.Binding)testResourceBindingListIterator.next();
			tempBoundMember1 = pb.getBoundMember();
			pb2 = (Folder.Binding)testResource2BindingListIterator.next();
			tempBoundMember2 = pb2.getBoundMember();
			assertEquals("Unexpected Bound Member.",
						 tempBoundMember1.location().string(),
						 tempBoundMember2.location().string());
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a resource and write some content in it.
	 * <li>Try to read the property "getcontentlength" of the created resource as
	 * an attribute (since that property is not a WVCM property).
	 * <li>Check that the result is neither null not empty (if default value
	 * is not empty).
	 *
	 * @throws Exception
	 */
	public void testGetNonWvcmWebdavPropertyAsAttribute() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		String propertyToRead = "getcontentlength";
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		
		AttributeName resxType = new AttributeName("DAV:", propertyToRead);
		PropertyNameList wantedprops = new PropertyNameList( new PropertyName[]{resxType} );
		
		testResource = testResource.doReadProperties(wantedprops);
		String contentLength = (String)testResource.getAttribute(resxType);
		assertNotNull("NON WVCM WEBDAV PROPERTY ("
						  + propertyToRead + ") RETURNED NULL.", contentLength);
		assertTrue("NON WVCM WEBDAV PROPERTY (" + propertyToRead
					   + ") RETURNED EMPTY UNEXPECTEDLY.",
				   contentLength.length() > 0);
		if (logger.isLoggable(Level.CONFIG))
			logger.config("\nDAV:" + propertyToRead + " resource: "
							  + testResource.getAttribute(resxType));
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Read a WebDAV property from the resource as an attribute.
	 * <li>Get properties of the resource by the name of the property.
	 * <li>Check that the properties are not null.
	 *
	 * @throws Exception
	 */
	public void testGetProperty() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		AttributeName resxTypeAttribute = new AttributeName("DAV:", "resourcetype");
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{resxTypeAttribute} );
		
		testResource = testResource.doReadProperties(wantedprops);
		PropertyName[] pns = testResource.getPropertyNameList(false).getPropertyNames();
		for ( int i = 0; i < pns.length; i++)
		{
			assertNotNull("PROPERTY " + pns[i].getString() + " RECEIVED IS NULL UNEXPECTEDLY.",
						  String.valueOf(testResource.getProperty(pns[i])));
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set a comment, persist it and get it again.
	 * <li>Check comment is correct in the resx.
	 *
	 * @throws Exception
	 */
	public void testSetGetComment() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String comment = "testSetGetComment comment";
		testResource.setComment(comment);
		
		testResource.doWriteProperties();
		
		assertEquals("Unexpected comment", comment, testResource.getComment());
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set its creator display name. Persist change.
	 * <li>Check property.
	 *
	 * @throws Exception
	 */
	// TODO: REMOVE THIS TEST/CHECK FOR OTHER TEST CASE.
	public void testSetGetCreatorDisplayName() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String creatorDisplayName = "testSetCreatorDisplayName creatorDisplayName";
		testResource.setCreatorDisplayName(creatorDisplayName);
		
		testResource.doWriteProperties();
		
		assertEquals("Unexpected CreatorDisplayName", creatorDisplayName, testResource.getCreatorDisplayName());
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set its display name. Persist change.
	 * <li>Check property.
	 *
	 * @throws Exception
	 */
	public void testSetGetDisplayName() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String displayName = "testSetGetDisplayName displayName";
		testResource.setDisplayName(displayName);
		
		testResource.doWriteProperties();
		
		assertEquals("Unexpected DisplayName", displayName, testResource.getDisplayName());
		
		
	}
	
	// TODO: ADD TESTS REGARDING TO CREATING RESOURCES WITHOUT A WISHED PROPERTY AND GETTING THOSE PROPERTIES.
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set a property to it and persist change.
	 * <li>Get the property and check the value received.
	 *
	 * @throws Exception
	 */
	public void testSetGetAttribute() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		String myAttributeValue = "My Attribute Value";
		testResource.setAttribute(att, myAttributeValue);
		
		testResource.doWriteProperties();
		
		assertEquals("Unexpected Attribute Value", myAttributeValue, testResource.getAttribute(att));
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get the provider list.
	 * <li>Check the values received (at least there should be one provider).
	 *
	 * @throws Exception
	 */
	public void testGetProviderList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		int noOfProvider = 0;
		PropertyNameList wantedProps = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.PROVIDER_LIST} );
		testResource = testResource.doReadProperties(wantedProps);
		try
		{
			List providerList = testResource.getProviderList();
			assertNotNull("Provider List returned is null???",
						  providerList);
			assertTrue("Provider List has no members unexpectedly.",
					   providerList.size() > 0);
			
			
		}
		
		catch (WvcmException we)
		{
			
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			System.out.println(ReasonCodeChecker.getReasonCodeAsString(we));
			fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
		
		assertTrue("No provider",noOfProvider> 0);
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get the principal folder list for the resx created.
	 * <li>Check the received list.
	 *
	 * @throws Exception
	 */
	// TODO: CHECK WHICH RESULTS SHOULD COME (WITH PETER?)
	public void testGetPrincipalFolderList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		// Read Principal_folder_list property
		PropertyNameList wantedProps = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.PRINCIPAL_FOLDER_LIST} );
		testResource = testResource.doReadProperties(wantedProps);
		String tempPrincipalFolderPath;
		try
		{
			List principalFolderList = testResource.getPrincipalFolderList();
			assertTrue("List should not be empty.",
					   principalFolderList.size() > 0);
			
			assertEquals("List size is unexpected.",
						 3,
						 principalFolderList.size());
			
			for (int i = 0; i< principalFolderList.size(); i++)
			{
				tempPrincipalFolderPath = principalFolderList.get(i).toString();
				assertTrue("Unexpected folder in principal folder list:\n" +
							   tempPrincipalFolderPath,
						   tempPrincipalFolderPath.endsWith(usersFolderPath) ||
							   tempPrincipalFolderPath.endsWith(groupsFolderPath) ||
							   tempPrincipalFolderPath.endsWith(rolesFolderPath));
			}
			
		}
		catch (WvcmException we)
		{
			
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			assertEquals("GetPrincipalFolderList fails", ReasonCode.VALUE_UNAVAILABLE, we.getReasonCode());
			//fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get the privilege folder list for the resx created.
	 * <li>Check the received list.
	 *
	 * @throws Exception
	 */
	// TODO: CHECK WHICH RESULTS SHOULD COME (WITH PETER?)
	public void testGetPrivilegeFolderList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedProps = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.PRIVILEGE_FOLDER_LIST} );
		testResource = testResource.doReadProperties(wantedProps);
		
		try
		{
			List privilegeFolderList = testResource.getPrivilegeFolderList();
			assertTrue("List should not be empty.",
					   privilegeFolderList.size() > 0);
			
			assertEquals("List size is unexpected.",
						 1,
						 privilegeFolderList.size());
			
			assertEquals("Unexpected result in privilege folder list result.",
						 contextPath() + "/" + privilegesFolderName,
						 privilegeFolderList.get(0).toString());
			
		}
		catch (WvcmException we)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			
			assertEquals("getSupportedFolderList fails", ReasonCode.VALUE_UNAVAILABLE, we.getReasonCode());
			//fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
	}
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Get the supported privilege list for the resx created.
	 * <li>Check the received list.
	 *
	 * @throws Exception
	 */
	// TODO: CHECK WHICH RESULTS SHOULD COME (WITH PETER?)
	public void testGetSupportedPrivilegeList() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		PropertyNameList wantedProps = new PropertyNameList(
			new PropertyName[]
			{
				PropertyName.SUPPORTED_PRIVILEGE_LIST} );
		testResource = testResource.doReadProperties(wantedProps);
		
		try
		{
			List privilegeList = testResource.getSupportedPrivilegeList();
			assertTrue("List should not be empty.",
					   privilegeList.size() > 0);
			
			for (int i=0; i<privilegeList.size(); i++)
			{
				System.out.println(privilegeList.get(i));
			}
			
		}
		catch (WvcmException we)
		{
			if(logger.isLoggable(Level.CONFIG))
				logger.config("Exception message:\n" + we.getMessage());
			assertEquals("getSupportedPrivilegeList fails", ReasonCode.VALUE_UNAVAILABLE, we.getReasonCode());
			//fail("Exception : " +ReasonCodeChecker.getReasonCodeAsString(we));
		}
		
	}
}










