/*
 * $Header: /home/cvspublic/jakarta-slide/proposals/wvcm/test/junit/src/org/apache/wvcm/test/i18n/ResxTest.java,v 1.3 2004/07/30 06:52:36 ozeigermann Exp $
 * $Revision: 1.3 $
 * $Date: 2004/07/30 06:52:36 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.wvcm.test.i18n;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.util.Locale;
import java.util.logging.Logger;
import javax.wvcm.PropertyNameList;
import javax.wvcm.PropertyNameList.AttributeName;
import javax.wvcm.PropertyNameList.PropertyName;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.apache.wvcm.test.common.AbstractTest;

/**
 * Functional tests for Resource.
 *
 * @author <a href="mailto:peter.nevermann@softwareag.com">Peter Nevermann</a>
 * @version $Revision: 1.3 $
 */
public class ResxTest extends AbstractTest
{
	
	private static Logger logger = Logger.getLogger("org.apache.wvcm.test.ResxTest");
    
    private String destinationPath = null;
    
	private String testResx1 = "test.xml";
	private String testResxCopy = "testCopy.xml";
	private String testHelloResx = "Hello.txt";
	
	private String testResourcesi18n = "i18n/";
	
    /**
	 * Default Constructor
	 *
	 * @param    testName            Name of the test to run.
	 *
	 * @throws   Exception
	 *
	 */
    public ResxTest(String testName) throws Exception
	{
		super(testName);
		resourcePath = rootFolderPath() + "/" + testResx1;
		destinationPath = rootFolderPath() + "/" + testResxCopy;
		
    }
    
    /**
	 * Creates a TestSuite with tests contained in this class.
	 *
	 * @return   a Test
	 */
    public static Test suite()
	{
		return new TestSuite(ResxTest.class);
    }
    
    /**
	 * Start test suite contained in this class using the text interface. If a
	 * parameter is passed, it will use it as a test name and run only that test
	 * case.
	 *
	 * @param args   commandline options (first argument = test method name to run)
	 *
	 * @throws Exception
	 */
    public static void main( String[] args ) throws Exception
	{
		if (args.length == 1)
		{
			TestSuite testSuite = new TestSuite();
			testSuite.addTest(new ResxTest(args[0]));
			junit.textui.TestRunner.run(testSuite);
		}
		else
		{
			junit.textui.TestRunner.run( suite() );
		}
    }
    
    /**
	 * Common test setup.
	 *
	 * @throws Exception
	 */
    protected void setUp() throws Exception
	{
		super.setUp ();
    }
    
    /**
	 * Tears down the fixture, for example, close a network connection.
	 * This method is called after a test is executed.
	 *
	 * @throws Exception
	 */
    protected void tearDown() throws Exception
	{
    }
	
	/**
	 * <ul>
	 * <li>Create a Resource.
	 * <li>Write some content in it.
	 * <li>Read content and compare with the written one.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoReadContentEn() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);

		String testFileName = testResourcesRoot
			+ testResourcesi18n
			+ "schema_en/UTF-8/Attribute/"
			+ "inst_table_attributei18n_utf-8NP.xml";
		
		File other = new File(testResourcesAbsoluteRoot + "\\" + testFileName);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		InputStream streamForTest = getClass().getResourceAsStream(testFileName);
		
		testResource.doWriteContent(
			streamForTest,
			cid );
		streamForTest.close();

		ByteArrayOutputStream contentToCompare = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(null, contentToCompare);

		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.compareXML(other.getAbsolutePath(),
										contentToCompare));
		
	}
	public void testDoReadContentJa() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);

		String testFileName = testResourcesRoot
			+ testResourcesi18n
			+ "schema_ja/SHIFT_JIS/subelement/"
			+ "inst_table_subelementi18n_Shift_JIS.xml";
		
		File other = new File(testResourcesAbsoluteRoot + "\\" + testFileName);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		InputStream streamForTest = getClass().getResourceAsStream(testFileName);
		
		testResource.doWriteContent(
			streamForTest,
			cid );
		streamForTest.close();

		ByteArrayOutputStream contentToCompare = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(null, contentToCompare);

		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.compareXML(other.getAbsolutePath(),
										contentToCompare));
		
	}
	public void testDoReadContentZh() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);

		String testFileName = testResourcesRoot
			+ testResourcesi18n
			+ "schema_zh/Big5/rootelement/"
			+ "inst3_table_doctypei18n_Big5JP.xml";
		
		File other = new File(testResourcesAbsoluteRoot + "\\" + testFileName);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		InputStream streamForTest = getClass().getResourceAsStream(testFileName);
		
		testResource.doWriteContent(
			streamForTest,
			cid );
		streamForTest.close();

		ByteArrayOutputStream contentToCompare = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(null, contentToCompare);

		assertTrue("Content of resource copied differs from original.",
				   xmlHelper.compareXML(other.getAbsolutePath(),
										contentToCompare));
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes on the resource and persist the changes.
	 * <li>Check attributes are there.
	 * <li>Remove one of the properties.
	 * <li>Read attributes again and check it is deleted.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testAttributeValues() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "att-1");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "att-2");
		AttributeName att3 = new AttributeName("http://my.namespace.org", "att-3");
		
		String att1Value = "\u00E0\u00E7\u00E8\u00E9\u00E4\u00F6\u00FC\u00C4\u00D6\u00DCutf8";
		String att2Value = "\u88FD\u54C1\u756A\u53F7utf8";
		String att3Value = "\u5373\u53EF\u986F\u793Autf8";
		
		testResource.setAttribute( att1, att1Value );
		testResource.setAttribute( att2, att2Value);
		testResource.setAttribute( att3, att3Value );
		testResource.doWriteProperties();
		
		assertEquals("Unexpected attribute 1 value in proxy",
					 att1Value,
					 testResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in proxy",
					 att2Value,
					 testResource.getAttribute(att2));
		assertEquals("Unexpected attribute 3 value in proxy",
					 att3Value,
					 testResource.getAttribute(att3));
		
		
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					att1,
						att2,
						att3} );
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, content );
		
		assertEquals("Unexpected attribute 1 value in Resource",
					 att1Value,
					 testResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in Resource",
					 att2Value,
					 testResource.getAttribute(att2));
		assertEquals("Unexpected attribute 3 value in Resource",
					 att3Value,
					 testResource.getAttribute(att3));
		
		
		
		PropertyNameList attnames = testResource.getPropertyNameList(true);
		int n = attnames.getPropertyNames().length;
		
		testResource.removeAttribute( att1 );
		testResource.doWriteProperties();
		
		attnames = testResource.getPropertyNameList(true);
		int m = attnames.getPropertyNames().length;
		
		assertTrue( m == (n - 1) );
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>set two valid attributes on the resource and persist the changes.
	 * <li>Check attributes are there.
	 * <li>Remove one of the properties.
	 * <li>Read attributes again and check it is deleted.
	 *
	 * @throws   Exception
	 *
	 */
	
	public void testAttributeNames() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att1 = new AttributeName("http://my.namespace.org", "\u00E0\u00E7\u00E8\u00E9\u00E4\u00F6\u00FC\u00C4\u00D6\u00DCutf8");
		AttributeName att2 = new AttributeName("http://my.namespace.org", "\u88FD\u54C1\u756A\u53F7utf8");
		AttributeName att3 = new AttributeName("http://my.namespace.org", "\u5373\u53EF\u986F\u793Autf8");
		
		String att1Value = "Western";
		String att2Value = "Japanese";
		String att3Value = "Chinese";
		
		testResource.setAttribute( att1, att1Value );
		testResource.setAttribute( att2, att2Value);
		testResource.setAttribute( att3, att3Value );
		testResource.doWriteProperties();
		
		assertEquals("Unexpected attribute 1 value in proxy",
					 att1Value,
					 testResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in proxy",
					 att2Value,
					 testResource.getAttribute(att2));
		assertEquals("Unexpected attribute 3 value in proxy",
					 att3Value,
					 testResource.getAttribute(att3));
		
		
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					att1,
						att2,
						att3} );
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent( wantedprops, content );
		
		assertEquals("Unexpected attribute 1 value in Resource",
					 att1Value,
					 testResource.getAttribute(att1));
		assertEquals("Unexpected attribute 2 value in Resource",
					 att2Value,
					 testResource.getAttribute(att2));
		assertEquals("Unexpected attribute 3 value in Resource",
					 att3Value,
					 testResource.getAttribute(att3));
		
		
		
		PropertyNameList attnames = testResource.getPropertyNameList(true);
		int n = attnames.getPropertyNames().length;
		
		testResource.removeAttribute( att1 );
		testResource.doWriteProperties();
		
		attnames = testResource.getPropertyNameList(true);
		int m = attnames.getPropertyNames().length;
		
		assertTrue( m == (n - 1) );
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Get property ContentLanguage.
	 * <li>Check that the property returned is not null or an empty string.
	 * <li>Set another content language in the resource.
	 * <li>Get againg property ContentLanguage.
	 * <li>Check that the property returned is equal to the set one.
	 *
	 * @throws   Exception
	 *
	 */
	public void testSetGetContentLanguage() throws Exception
	{
		location = provider.location( resourcePath );
		testResource= createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList( new PropertyName[]
															{
					PropertyName.CONTENT_LANGUAGE} );
		
		testResource = testResource.doReadProperties(wantedprops);
		Locale contentLanguage = testResource.getContentLanguage();
		assertTrue("Content language not received as expected.",
					   (contentLanguage != null) && (contentLanguage.toString().length()>0));
		
		Locale testContentLanguage = Locale.ITALIAN;
		testResource.setContentLanguage(testContentLanguage);
		testResource.doWriteProperties();
		testResource = testResource.doReadProperties(wantedprops);
		assertEquals("Content language not received as expected.",
					 testContentLanguage,
					 testResource.getContentLanguage());
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Get property Content Character Set.
	 * <li>Check that the property returned is not null or an empty string.
	 *
	 * @throws   Exception
	 *
	 */
	// TODO: CHECK WHY IT RETURNS A VALUE ONLY AFTER A READ CONTENT...
	// TODO: ADD SET CONTENT CHARACTER SET PART. REMOVE PART FROM OTHER TEST CASE.
	public void testSetGetContentCharacterSet() throws Exception
	{
		location = provider.location( resourcePath );
		testResource= createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + "test.xml"),
			cid );
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.CONTENT_CHARACTER_SET} );
		
		//testResource = testResource.doReadProperties(wantedprops);
		
		ByteArrayOutputStream content = new ByteArrayOutputStream();
		testResource = testResource.doReadContent(wantedprops, content);
		
		String contentCharSet = testResource.getContentCharacterSet();
		assertTrue("Unexpected character set.",
				   contentCharSet != null && contentCharSet.length() > 0);
	}
	
	
	
	/**
	 * <ul>
	 * <li>Create and write content in a Resource.
	 * <li>Write some properties in the resource.
	 * <li>Check updated property list contains those props.
	 * <li>DoWriteProperties (makes properties persistent in server).
	 * <li>Check updated property list is empty.
	 * <li>Check that the properties set is correct.
	 *
	 * @throws   Exception
	 *
	 */
	public void testDoWriteProperties() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		AttributeName att = new AttributeName("http://my.namespace.org", "myAttribute");
		testResource.setComment( "My Comment" );
		testResource.setDisplayName( "My Display Name" );
		testResource.setContentLanguage( new Locale("de","") );
		testResource.setAttribute( att, "My Attribute Value" );
		
		testResource.doWriteProperties();
		
		PropertyNameList wantedprops =
			new PropertyNameList( new PropertyName[]
								 {
					PropertyName.COMMENT,
						PropertyName.DISPLAY_NAME,
						PropertyName.CONTENT_LANGUAGE,
						att} );
		testResource = testResource.doReadProperties(wantedprops);
		assertEquals( "My Comment", testResource.getComment() );
		assertEquals( "My Display Name", testResource.getDisplayName() );
		assertEquals( "de", testResource.getContentLanguage().toString() );
		assertEquals( "My Attribute Value", testResource.getAttribute(att) );
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Read a WebDAV property from the resource as an attribute.
	 * <li>Get properties of the resource by the name of the property.
	 * <li>Check that the properties are not null.
	 *
	 * @throws Exception
	 */
	public void testGetProperty() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String cid = propsHelper.getContentIdentifier(testResource);
		testResource.doWriteContent(
			getClass().getResourceAsStream(testResourcesRoot + testResx1),
			cid );
		
		AttributeName resxTypeAttribute = new AttributeName("DAV:", "resourcetype");
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{resxTypeAttribute} );
		
		testResource = testResource.doReadProperties(wantedprops);
		PropertyName[] pns = testResource.getPropertyNameList(false).getPropertyNames();
		for ( int i = 0; i < pns.length; i++)
		{
			assertNotNull("PROPERTY " + pns[i].getString() + " RECEIVED IS NULL UNEXPECTEDLY.",
						  String.valueOf(testResource.getProperty(pns[i])));
		}
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set a comment, persist it and get it again.
	 * <li>Check comment is correct in the resx.
	 *
	 * @throws Exception
	 */
	public void testSetGetComment() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String comment = "testSetGetComment \u88FD\u54C1\u756A\u53F7shiftjis";
		testResource.setComment(comment);
		
		assertEquals("Unexpected comment in proxy", comment, testResource.getComment());
		
		testResource.doWriteProperties();
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.COMMENT} );
		
		testResource = testResource.doReadProperties(wantedprops);
		
		assertEquals("Unexpected comment in resource", comment, testResource.getComment());
		
	}
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set its creator display name. Persist change.
	 * <li>Check property.
	 *
	 * @throws Exception
	 */
	
	public void testSetGetCreatorDisplayName() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String creatorDisplayName = "testSetCreatorDisplayName \u5373\u53EF\u986F\u793Abig5";
		testResource.setCreatorDisplayName(creatorDisplayName);
		
		assertEquals("Unexpected CreatorDisplayName in proxy", creatorDisplayName, testResource.getCreatorDisplayName());
		
		testResource.doWriteProperties();
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.CREATOR_DISPLAY_NAME} );
		
		testResource = testResource.doReadProperties(wantedprops);
		
		assertEquals("Unexpected CreatorDisplayName in resource", creatorDisplayName, testResource.getCreatorDisplayName());
		
	}
	
	
	/**
	 * <ul>
	 * <li>Create a resource.
	 * <li>Set its display name. Persist change.
	 * <li>Check property.
	 *
	 * @throws Exception
	 */
	public void testSetGetDisplayName() throws Exception
	{
		location = provider.location( resourcePath );
		testResource = createControllableResource(location);
		
		String displayName = "testSetGetDisplayName \u00E0\u00E7\u00E8\u00E9\u00E4\u00F6\u00FC\u00C4\u00D6\u00DCutf8";
		testResource.setDisplayName(displayName);
		
		assertEquals("Unexpected DisplayName in proxy", displayName, testResource.getDisplayName());
		
		testResource.doWriteProperties();
		
		PropertyNameList wantedprops = new PropertyNameList(
			new PropertyName[]{PropertyName.DISPLAY_NAME} );
		
		testResource = testResource.doReadProperties(wantedprops);
		
		assertEquals("Unexpected DisplayName in Resource", displayName, testResource.getDisplayName());
		
		
	}
	
}

