/*
 * $Header: /home/cvspublic/jakarta-slide/src/share/org/apache/slide/search/basic/QueryTree.java,v 1.6 2004/12/07 17:51:39 luetzkendorf Exp $
 * $Revision: 1.6 $
 * $Date: 2004/12/07 17:51:39 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.search.basic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.slide.common.Scope;
import org.apache.slide.search.InvalidScopeException;

/**
 * Represents the scopes for all stores, that are within the scope of one query.
 *
 * @version $Revision: 1.6 $
 *
 **/
public class QueryTree {
    
    private TokenizedScope topNode;
    
    private List allQueryTreeNodes = new ArrayList ();
    
    
    //    QueryTree (Enumeration stores, Scope scope)
    //        throws InvalidScopeException
    //    {
    //        this (stores, scope, new Scope [0]);
    //    }
    
    /**
     * Constructs a query tree
     *
     * @param stores  all stores, that are within this namespace
     * @param scope   the scope of this query
     * @param excluded a list of scopes, that shall be excluded from search
     */
    QueryTree (Enumeration stores, Scope scope, Scope [] excluded)
        throws InvalidScopeException
    {
        // System.out.println (scope);
        topNode = new TokenizedScope (scope);
        
        while (stores.hasMoreElements()) {
            Scope configuredStore = (Scope)stores.nextElement();
            TokenizedScope tConfStore = new TokenizedScope (configuredStore);
            if (tConfStore.isChildOf (topNode) && tConfStore.isNotExcluded (excluded) ) {
                allQueryTreeNodes.add (configuredStore);
            }
        }
        allQueryTreeNodes.add (scope);
        
    }
    
    /**
     * Checks, if the indicated scope has children within this QueryTree.
     *
     * @param    scopeToBeChecked    the Scope to be checked
     *
     * @return   a boolean
     *
     */
    public boolean hasChildren (Scope scopeToBeChecked) {
        TokenizedScope tScopeToBeChecked = new TokenizedScope (scopeToBeChecked);
        Iterator it = allQueryTreeNodes.iterator();
        while (it.hasNext()) {
            TokenizedScope ts = new TokenizedScope ((Scope)it.next());
            if (ts.isChildOf (tScopeToBeChecked))
                return true;
        }
        
        return false;
    }
    
    /**
     * Returns a set of all scopes in the query that are children of 
     * the <code>scopeToBeChecked</code>.
     * @param scopeToBeChecked
     */
    public Set getChildren(Scope scopeToBeChecked) {
        Set result = null;
        
        TokenizedScope tScopeToBeChecked = new TokenizedScope (scopeToBeChecked);
        Iterator it = allQueryTreeNodes.iterator();
        while (it.hasNext()) {
            Scope scope = (Scope)it.next();
            TokenizedScope ts = new TokenizedScope(scope);
            if (ts.isChildOf (tScopeToBeChecked)) {
                if (result == null) result = new HashSet();
                result.add(scope);
            }
        }
        
        return result != null ? result : Collections.EMPTY_SET;
    }
    
    /**
     * calculates the depth of scope within this QueryTree (relative to the
     * topLevel of tree)
     *
     * @param    scope               a  Scope
     *
     * @return   an int
     *
     */
    public int relativeDepth (Scope scope) {
        TokenizedScope tScope = new TokenizedScope (scope);
        //boolean contains = allQueryTreeNodes.contains (tScope);
        
        return tScope.depth - topNode.depth;
    }
    
    
    /**
     * Retrieves an iterator of all scopes in this Tree
     *
     * @return   an Iterator of Scope objects
     *
     */
    public Iterator iterator () {
        return allQueryTreeNodes.iterator();
    }
    
    
    /**
     * Helper class to handle the scopes
     *
     * @version $Revision: 1.6 $
     *
     **/
    class TokenizedScope {
        int depth;
        Scope scope;
        
        TokenizedScope (Scope   scope) {
            this.scope = scope;
            depth = (new StringTokenizer (scope.toString(), "/")).countTokens();
        }
        
        /**
         * Method equals
         *
         * @param    o                   an Object
         *
         * @return   a boolean
         *
         */
        public boolean equals (Object o) {
            return scope.equals(o);
        }
        
        /**
         * Method toString
         *
         * @return   a String
         *
         */
        public String toString () {
            return scope.toString();
        }
        
        /**
         * Method isChildOf
         *
         * @param    tScope              a  TokenizedScope
         *
         * @return   a boolean
         *
         */
        public boolean isChildOf (TokenizedScope tScope) {
            if (depth > tScope.depth && toString().startsWith(tScope.toString())) {
                return true;
            }
            else {
                return false;
            }
        }
        
        /**
         * Method isNotExcluded
         *
         * @param    excluded            a  Scope[]
         *
         * @return   a boolean
         *
         */
        public boolean isNotExcluded (Scope []excluded) {
            for (int i = 0; i < excluded.length; i++) {
                if (this.toString().startsWith(excluded [i].toString())) {
                    return false;
                }
            }
            return true;
        }
    }
}


