/*
 * $Header: /home/cvspublic/jakarta-slide/src/share/org/apache/slide/store/MacroStore.java,v 1.2 2004/10/29 13:32:03 ozeigermann Exp $
 * $Revision: 1.2 $
 * $Date: 2004/10/29 13:32:03 $
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.store;

import org.apache.slide.common.Service;
import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.common.Uri;
import org.apache.slide.structure.ObjectAlreadyExistsException;
import org.apache.slide.structure.ObjectNotFoundException;

/**
 * Store for macro opertations support. This store executes compound operations
 * on hierarchical data. A recursive copy, a recursive delete and a recursive
 * move resp. rename are supported provided the {@link #isMacroCopySupported()},
 * {@link #isMacroMoveSupported()}and/or {@link #isMacroDeleteSupported()}
 * indicate this. By having checks for all three macro opertations stores can
 * dynamically decide which kind of operations they support. E.g. This allows
 * for stores that only support recursive deletes.
 * 
 * @version $Revision: 1.2 $
 */
public interface MacroStore extends Service {

    /**
     * Checks if this store instance actually supports macro delete operation.
     * It may seem clear this store supports sequences as it implements this
     * interface, but a request to the underlying persistence store might be
     * needed to dynamically find out.
     * 
     * @return <code>true</code> if the store supports the macro delete
     *         operation, <code>false</code> otherwise
     */
    public boolean isMacroDeleteSupported();

    /**
     * Deletes an object recursively.
     * 
     * @param targetUri
     *            Uri of the object to delete
     * @throws ObjectNotFoundException if the object to delete was not found
     * @throws ServiceAccessException
     *             if anything else goes wrong while deleting the object
     */
    public void macroDelete(Uri targetUri) throws ServiceAccessException, ObjectNotFoundException;

    /**
     * Checks if this store instance actually supports macro copy operation. It
     * may seem clear this store supports sequences as it implements this
     * interface, but a request to the underlying persistence store might be
     * needed to dynamically find out.
     * 
     * @return <code>true</code> if the store supports the macro copy
     *         operation, <code>false</code> otherwise
     */
    public boolean isMacroCopySupported();

    /**
     * Recursively copies an object.
     * 
     * @param sourceUri the source URI of the copy
     * @param targetUri the destination URI of the copy
     * @param overwrite <code>true</code> if the target shall be copied to even if it already exsis
     * @param recursive <code>true</code> if all descendants of the source should be copied as well
     * @throws ObjectNotFoundException if the object to copy from was not found
     * @throws ObjectAlreadyExistsException if the object to copy to to was already there
     * @throws ServiceAccessException
     *             if anything else goes wrong while copying the object
     */
    public void macroCopy(Uri sourceUri, Uri targetUri, boolean overwrite, boolean recursive) throws ServiceAccessException, ObjectNotFoundException,
            ObjectAlreadyExistsException;

    /**
     * Checks if this store instance actually supports macro move operation. It
     * may seem clear this store supports sequences as it implements this
     * interface, but a request to the underlying persistence store might be
     * needed to dynamically find out.
     * 
     * @return <code>true</code> if the store supports the macro move
     *         operation, <code>false</code> otherwise
     */
    public boolean isMacroMoveSupported();

    /**
     * Recursively moves an object.
     * 
     * @param sourceUri the source URI of the move
     * @param targetUri the destination URI of the move
     * @param overwrite <code>true</code> if the target shall be copied to even if it already exsis
     * @throws ObjectNotFoundException if the object to move from was not found
     * @throws ObjectAlreadyExistsException if the object to move to to was already there
     * @throws ServiceAccessException
     *             if anything else goes wrong while moving the object
     */
    public void macroMove(Uri sourceUri, Uri targetUri, boolean overwrite) throws ServiceAccessException, ObjectNotFoundException,
            ObjectAlreadyExistsException;

}