/*
 * $Header: /home/cvspublic/jakarta-slide/src/stores/org/apache/slide/index/lucene/expressions/GtExpression.java,v 1.7 2004/12/07 17:49:57 luetzkendorf Exp $
 * $Revision: 1.7 $
 * $Date: 2004/12/07 17:49:57 $
 *
 * ====================================================================
 *
 * Copyright 1999-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.index.lucene.expressions;

import java.util.Date;

import org.apache.lucene.index.Term;
import org.apache.lucene.search.RangeQuery;

import org.apache.slide.index.lucene.Index;
import org.apache.slide.index.lucene.IndexConfiguration;
import org.apache.slide.search.BadQueryException;
import org.jdom.Element;

/**
 * Implements <code>gt</code> and <code>gte</code> expression.
 */
public class GtExpression extends AbstractLuceneExpression implements RangeOperator
{
    private String field;
    private String value;
    private boolean inclusive;

    public GtExpression(Index index, Element element, boolean inclusive)
            throws BadQueryException
    {
        super(index);
        this.inclusive = inclusive;

        IndexConfiguration config = index.getConfiguration();
        Element prop = getPropertyElement(element);
        this.field = config.generateFieldName(prop.getNamespaceURI(), prop.getName());
        Element literal = getLiteralElement(element);

        String upperBound;
        if (index.getConfiguration().isDateProperty(prop.getNamespaceURI(), prop.getName())) {
            Date date = IndexConfiguration.getDateValue(literal.getTextTrim());
            this.value = config.dateToIndexString(date);
            upperBound = Index.DATE_UPPER_BOUND;
        } else if (index.getConfiguration().isIntProperty(prop.getNamespaceURI(), prop.getName())) {
            this.value = config.intToIndexString(Long.parseLong(literal.getTextTrim()));
            upperBound = Index.INT_UPPER_BOUND;
        } else {
            this.value = literal.getTextTrim();
            upperBound = Index.STRING_UPPER_BOUND;
        }

        RangeQuery rangeQuery = new RangeQuery(
                new Term(this.field, this.value),
                new Term(this.field, upperBound), 
                inclusive); // inclusive or not

        setQuery(rangeQuery);
    }
    
    public String getField() {
        return this.field;
    }
    
    public String getValue() {
        return this.value;
    }
    public boolean inclusive() {
        return this.inclusive;
    }
    
    public String toString() {
        StringBuffer b = new StringBuffer();
        
        b.append('(')
         .append(this.field)
         .append('>')
         .append(this.inclusive ? "=" : "")
         .append(this.value)
         .append(')');
        
        return b.toString();
    }
}