/*
 * $Header: /home/cvspublic/jakarta-slide/src/stores/org/apache/slide/store/file/AbstractSimpleStore.java,v 1.3 2005/01/03 15:59:53 luetzkendorf Exp $
 * $Revision: 1.3 $
 * $Date: 2005/01/03 15:59:53 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.store.file;

import java.util.*;
import javax.transaction.xa.*;
import org.apache.slide.common.*;
import org.apache.slide.store.*;
import org.apache.slide.content.*;
import org.apache.slide.structure.*;
import org.apache.slide.util.logger.*;

/**
 * AbstractSimpleStore provides a minimal implementation to support creating a Slide Store.
 * It no-ops most methods and leaves abstract those methods you must implement to support
 * basic create/retrieve/update/remove.
 * <p>
 * For locking and security your implementation should be paired with
 * org.apache.slide.store.mem.TransientSecurityStore and
 * org.apache.slide.store.mem.TransientLockStore.
 *
 * @see SimpleFileStore
 * @author Alon Salant
 */
public abstract class AbstractSimpleStore
  extends AbstractServiceBase
  implements ContentStore, NodeStore, RevisionDescriptorStore, RevisionDescriptorsStore
{

  // ==== Service Methods ================================

  public void setParameters(Hashtable parameters)
    throws ServiceParameterErrorException, ServiceParameterMissingException
  {
    log("setParameters(" + parameters + ")");
  }

  public void connect() throws ServiceConnectionFailedException
  {
  }

  public void reset() throws ServiceResetFailedException
  {
  }

  public void disconnect() throws ServiceDisconnectionFailedException
  {
  }

  public boolean isConnected() throws ServiceAccessException
  {
    return true;
  }

  public int getTransactionTimeout() throws XAException
  {
    return 0;
  }

  public boolean setTransactionTimeout(int seconds) throws XAException
  {
    return false;
  }

  public boolean isSameRM(XAResource rm) throws XAException
  {
    return false;
  }

  public Xid[] recover(int flag) throws XAException
  {
    return new Xid[0];
  }

  public int prepare(Xid txId) throws XAException
  {
    return XA_RDONLY;
  }

  public void forget(Xid txId) throws XAException
  {
  }

  public void rollback(Xid txId) throws XAException
  {
  }

  public void end(Xid txId, int flags) throws XAException
  {
  }

  public void start(Xid txId, int flags) throws XAException
  {
  }

  public void commit(Xid txId, boolean onePhase) throws XAException
  {
  }

  // ==== ContentStore Methods ================================

  public abstract NodeRevisionContent retrieveRevisionContent(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException, RevisionNotFoundException;

  public abstract void createRevisionContent(Uri uri,
                                             NodeRevisionDescriptor revisionDescriptor,
                                             NodeRevisionContent revisionContent)
    throws ServiceAccessException, RevisionAlreadyExistException;

  public abstract void storeRevisionContent(Uri uri,
                                            NodeRevisionDescriptor revisionDescriptor,
                                            NodeRevisionContent revisionContent)
    throws ServiceAccessException, RevisionNotFoundException;

  public void removeRevisionContent(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException
  {
    log("removeRevisionContent(" + uri + ")");
  }

  // ==== NodeStore Methods ================================

  public abstract void storeObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectNotFoundException;

  public abstract void createObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectAlreadyExistsException;

  public abstract void removeObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectNotFoundException;

  public abstract ObjectNode retrieveObject(Uri uri)
    throws ServiceAccessException, ObjectNotFoundException;

  // ==== RevisionDescriptorsStore Methods ================================

  public NodeRevisionDescriptors retrieveRevisionDescriptors(Uri uri)
    throws ServiceAccessException, RevisionDescriptorNotFoundException
  {
    log("retrieveRevisionDescriptors(" + uri + ")");

    NodeRevisionNumber rev = new NodeRevisionNumber(1, 0);

    Hashtable workingRevisions = new Hashtable();
    workingRevisions.put("main", rev);

    Hashtable latestRevisionNumbers = new Hashtable();
    latestRevisionNumbers.put("main", rev);

	

    Hashtable branches = new Hashtable();
    branches.put(rev, new Vector());

    return new NodeRevisionDescriptors(uri.toString(),
                                       rev,
                                       workingRevisions,
                                       latestRevisionNumbers,
                                       branches,
                                       false);
  }

  public void createRevisionDescriptors(Uri uri, NodeRevisionDescriptors revisionDescriptors)
    throws ServiceAccessException
  {
    log("createRevisionDescriptors(" + uri + ")");
  }

  public void storeRevisionDescriptors(Uri uri, NodeRevisionDescriptors revisionDescriptors)
    throws ServiceAccessException, RevisionDescriptorNotFoundException
  {
    log("storeRevisionDescriptors(" + uri + ")");
  }

  public void removeRevisionDescriptors(Uri uri) throws ServiceAccessException
  {
    log("removeRevisionDescriptors(" + uri + ")");
  }

  // ==== RevisionDescriptorStore Methods ================================

  public void createRevisionDescriptor(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException
  {
    log("createRevisionDescriptor(" + uri + ")");
  }

  public void storeRevisionDescriptor(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException, RevisionDescriptorNotFoundException
  {
    log("storeRevisionDescriptor(" + uri + ")");
  }

  public void removeRevisionDescriptor(Uri uri, NodeRevisionNumber revisionNumber)
    throws ServiceAccessException
  {
    log("removeRevisionDescriptor(" + uri + ")");
  }

  public abstract NodeRevisionDescriptor retrieveRevisionDescriptor(Uri uri, NodeRevisionNumber revisionNumber)
    throws ServiceAccessException, RevisionDescriptorNotFoundException;

  protected void log(String msg)
  {
    if (getLogger() == null)
      System.out.println(this.getClass().getName() + ": " + msg);
    else
      getLogger().log(msg, this.getClass().getName(), Logger.DEBUG);
  }

}
