/*
 * $Header: /home/cvspublic/jakarta-slide/src/stores/org/apache/slide/store/file/CountdownStore.java,v 1.3 2005/01/03 15:59:53 luetzkendorf Exp $
 * $Revision: 1.3 $
 * $Date: 2005/01/03 15:59:53 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.store.file;

import java.util.Date;

import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.common.Uri;
import org.apache.slide.content.NodeRevisionContent;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.content.NodeRevisionNumber;
import org.apache.slide.content.RevisionAlreadyExistException;
import org.apache.slide.content.RevisionDescriptorNotFoundException;
import org.apache.slide.content.RevisionNotFoundException;
import org.apache.slide.structure.ObjectAlreadyExistsException;
import org.apache.slide.structure.ObjectNode;
import org.apache.slide.structure.ObjectNotFoundException;
import org.apache.slide.structure.SubjectNode;

/**
 * CountdownStore is an AbstractSimpleStore implementation that shows how
 * to support the Store APIs with a little bit of code.
 * <p>
 * It presents nested directories 10 .. 1 and a file 0.txt in
 * the directory 1. This structure is defined in code. You can get some
 * funky things to happen by trying to change this structure through a
 * webdav client since this implementation seems to support all
 * operations but does not preserve state.
 *
 * @author Alon Salant
 */
public class CountdownStore
  extends AbstractSimpleStore
{

  // ==== ContentStore Methods ================================

  public NodeRevisionContent retrieveRevisionContent(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException, RevisionNotFoundException
  {
    log("retrieveRevisionContent(" + uri + ")");
    NodeRevisionContent nrc = new NodeRevisionContent();
    nrc.setContent("blast off!!".getBytes());
    return nrc;
  }

  public void createRevisionContent(Uri uri,
                                    NodeRevisionDescriptor revisionDescriptor,
                                    NodeRevisionContent revisionContent)
    throws ServiceAccessException, RevisionAlreadyExistException
  {
    log("createRevisionContent(" + uri + ")");
  }

  public void storeRevisionContent(Uri uri,
                                   NodeRevisionDescriptor revisionDescriptor,
                                   NodeRevisionContent revisionContent)
    throws ServiceAccessException, RevisionNotFoundException
  {
    log("storeRevisionContent(" + uri + ")");
  }

  public void removeRevisionContent(Uri uri, NodeRevisionDescriptor revisionDescriptor)
    throws ServiceAccessException
  {
    log("removeRevisionContent(" + uri + ")");
  }

  // ==== NodeStore Methods ================================

  public void storeObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectNotFoundException
  {
    log("storeObject(" + uri + ")");
  }

  public void createObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectAlreadyExistsException
  {
    log("createObject(" + uri + ")");
  }

  public void removeObject(Uri uri, ObjectNode object)
    throws ServiceAccessException, ObjectNotFoundException
  {
    log("removeObject(" + uri + ")");
  }

  public ObjectNode retrieveObject(Uri uri)
    throws ServiceAccessException, ObjectNotFoundException
  {
    log("retrieveObject(" + uri + ")");

    SubjectNode subject = new SubjectNode(uri.toString());
    if (uri.toString().endsWith("0.txt")) return subject;

    if (uri.isStoreRoot())
    {
      subject.addChild(new SubjectNode("9"));
      return subject;
    }
    int count = Integer.parseInt(uri.toString().substring(uri.toString().length() - 1));
    if (count > 1)
    {
      subject.addChild(new SubjectNode(count - 1 + ""));
      return subject;
    }
    subject.addChild(new SubjectNode("0.txt"));
    return subject;
  }

  // ==== RevisionDescriptorStore Methods ================================

  public NodeRevisionDescriptor retrieveRevisionDescriptor(Uri uri,
                                                           NodeRevisionNumber revisionNumber) throws ServiceAccessException,
    RevisionDescriptorNotFoundException
  {
    log("retrieveRevisionDescriptor(" + uri + ")");

    NodeRevisionDescriptor descriptor = new NodeRevisionDescriptor(new NodeRevisionNumber(1, 0),
                                                                   NodeRevisionDescriptors.MAIN_BRANCH);
    descriptor.setCreationDate(new Date(0));
    descriptor.setLastModified(new Date(new Date().getTime() - 2000));
    descriptor.setModificationDate(descriptor.getLastModifiedAsDate());
    if (uri.toString().endsWith("0.txt"))
    {
      descriptor.setContentLength("blast off!!".getBytes().length);
    }
    else
    {
      descriptor.setResourceType("<collection/>");
    }
    return descriptor;
  }

}