/*
 * $Header: /home/cvspublic/jakarta-slide/src/stores/org/apache/slide/store/impl/rdbms/MySqlRDBMSAdapter.java,v 1.14 2004/12/15 16:50:27 ozeigermann Exp $
 * $Revision: 1.14 $
 * $Date: 2004/12/15 16:50:27 $
 *
 * ====================================================================
 *
 * Copyright 1999-2003 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.store.impl.rdbms;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.Enumeration;
import java.util.Iterator;
import java.util.Set;

import org.apache.slide.content.*;
import org.apache.slide.common.*;
import org.apache.slide.macro.ConflictException;
import org.apache.slide.store.ConcurrencyConflictError;
import org.apache.slide.structure.ObjectNotFoundException;
import org.apache.slide.util.logger.Logger;

/**
 * Adapter for MySQL 4.
 * 
 * @version $Revision: 1.14 $
 */

public class MySqlRDBMSAdapter extends StandardRDBMSAdapter implements SequenceAdapter {

    protected static final String LOG_CHANNEL = MySqlRDBMSAdapter.class.getName();

    protected static String normalizeSequenceName(String sequenceName) {
        return sequenceName.replace('-', '_').toUpperCase() + "_SEQ";
    }

    public MySqlRDBMSAdapter(Service service, Logger logger) {
        super(service, logger);
    }

    protected ServiceAccessException createException(SQLException e, String uri) {

        switch (e.getErrorCode()) {
            case 1213 : // thread was deadlock victim
                getLogger().log(e.getErrorCode() + ": Deadlock resolved on " + uri, LOG_CHANNEL, Logger.WARNING);
//              return new ServiceAccessException(this, new ConflictException(uri));
                throw new ConcurrencyConflictError(e, uri);

            default :
                return super.createException(e, uri);
        }
    }

    protected String convertRevisionNumberToComparable(String revisioNumber) {

        return "convert(SUBSTRING_INDEX("
            + revisioNumber
            + ", '.', 1), unsigned), convert(SUBSTRING_INDEX("
            + revisioNumber
            + ", '.', -1), unsigned)";
    }

    public boolean isSequenceSupported(Connection conn) {
        return true;
    }

    public boolean createSequence(Connection conn, String sequenceName) throws ServiceAccessException {

        String query = "CREATE TABLE " + normalizeSequenceName(sequenceName) + "(id INT auto_increment NOT NULL,PRIMARY KEY (id))";

        PreparedStatement statement = null;

        try {
            statement = conn.prepareStatement(query);
            statement.executeUpdate();
            return true;
        } catch (SQLException e) {
            throw new ServiceAccessException(service, e);
        } finally {
            close(statement);
        }

    }

    public long nextSequenceValue(Connection conn, String sequenceName) throws ServiceAccessException {
        String query = "INSERT INTO " + normalizeSequenceName(sequenceName) + " VALUES(0)";

        String selectQuery = "SELECT LAST_INSERT_ID()";

        PreparedStatement statement = null;
        PreparedStatement selectStatement = null;
        ResultSet res = null;

        try {
            statement = conn.prepareStatement(query);
            statement.executeUpdate();

            selectStatement = conn.prepareStatement(selectQuery);
            res = selectStatement.executeQuery();
            if (!res.next()) {
                throw new ServiceAccessException(service, "Could not increment sequence " + sequenceName);
            }
            long value = res.getLong(1);
            return value;
        } catch (SQLException e) {
            throw new ServiceAccessException(service, e);
        } finally {
            close(statement);
            close(selectStatement, res);
        }
    }

    public boolean sequenceExists(Connection conn, String sequenceName) throws ServiceAccessException {

        PreparedStatement selectStatement = null;
        ResultSet res = null;

        try {
            selectStatement = conn.prepareStatement("SELECT * FROM " + normalizeSequenceName(sequenceName));
            res = selectStatement.executeQuery();
            return true;
        } catch (SQLException e) {
            return false;
        } finally {
            close(selectStatement, res);
        }
    }
    public void storeRevisionDescriptor(Connection connection, Uri uri, NodeRevisionDescriptor revisionDescriptor)
        throws ServiceAccessException, RevisionDescriptorNotFoundException {
        PreparedStatement statement = null;
        try {
            removeVersionLabels(connection, uri, revisionDescriptor.getRevisionNumber());
            createVersionLabels(connection, uri, revisionDescriptor);
            String revisionNumber = revisionDescriptor.getRevisionNumber().toString();
            for (Enumeration properties = revisionDescriptor.enumerateRemovedProperties(); properties.hasMoreElements();) {
                try {
                    NodeProperty property = (NodeProperty) properties.nextElement();
                    statement = connection
                            .prepareStatement(deleteStatement("PROPERTIES","p",", VERSION_HISTORY vh, URI u where p.VERSION_ID = vh.VERSION_ID and vh.URI_ID = u.URI_ID AND u.URI_STRING = ? AND p.PROPERTY_NAME = ? AND p.PROPERTY_NAMESPACE = ? and vh.REVISION_NO = ?"));
                    statement.setString(1, uri.toString());
                    statement.setString(2, property.getName());
                    statement.setString(3, property.getNamespace());
                    statement.setString(4, revisionNumber);
                    statement.executeUpdate();
                } finally {
                    close(statement);
                }
            }

            for (Enumeration properties = revisionDescriptor.enumerateUpdatedProperties(); properties.hasMoreElements();) {
                NodeProperty property = (NodeProperty) properties.nextElement();
                int protectedProperty = property.isProtected() ? 1 : 0;
                try {
                    statement = connection
                            .prepareStatement(deleteStatement("PROPERTIES","p",", VERSION_HISTORY vh, URI u where p.VERSION_ID = vh.VERSION_ID and vh.URI_ID = u.URI_ID AND u.URI_STRING = ? AND p.PROPERTY_NAME = ? AND p.PROPERTY_NAMESPACE = ? and vh.REVISION_NO = ?"));
                    statement.setString(1, uri.toString());
                    statement.setString(2, property.getName());
                    statement.setString(3, property.getNamespace());
                    statement.setString(4, revisionNumber);
                    statement.executeUpdate();
                } finally {
                    close(statement);
                }
				try {
					statement = connection
							.prepareStatement("insert into PROPERTIES (VERSION_ID,PROPERTY_NAMESPACE,PROPERTY_NAME,PROPERTY_VALUE,PROPERTY_TYPE,IS_PROTECTED) select vh.VERSION_ID, ?, ?, ?, ?, ? from VERSION_HISTORY vh, URI u where vh.URI_ID = u.URI_ID and u.URI_STRING = ? and vh.REVISION_NO = ?");
					statement.setString(1, property.getNamespace());
					statement.setString(2, property.getName());
					statement.setString(3, property.getValue().toString());
					statement.setString(4, property.getType());
					statement.setInt(5, protectedProperty);
					statement.setString(6, uri.toString());
					statement.setString(7, revisionNumber);
					statement.executeUpdate();
				} finally {
					close(statement);
				}
            }
        } catch (SQLException e) {
            throw createException(e, uri.toString());
        }
        revisionDescriptor.resetUpdatedProperties();
        revisionDescriptor.resetRemovedProperties();
    }
    
    protected void clearBinding(Connection connection, Uri uri, Set updatedBindings)
        throws ServiceAccessException, ObjectNotFoundException, SQLException {
        PreparedStatement statement = null;

        // clear this uri from having bindings and being bound 
        int bsize = updatedBindings.size();
        //If there are bindings to update, only remove those from the database
        if (bsize > 0) {

            try {
                String bindings = "(";
                for (int i = 1; i < bsize; i++)
                    bindings = bindings + "?,";
                bindings = bindings + "?)";
                statement =
                    connection.prepareStatement(
                        deleteStatement("BINDING","c",", URI u where u.URI_ID = c.CHILD_UURI_ID and u.URI_STRING IN "
                            + bindings) );

                Iterator iter = updatedBindings.iterator();
                for (int i = 1; iter.hasNext(); i++) {
                    String next = iter.next().toString();
                    statement.setString(i, next);
                }
                statement.executeUpdate();
            } finally {
                close(statement);
            }

            try {
                statement =
                    connection.prepareStatement(
                        deleteStatement("PARENT_BINDING","c",", URI u where c.URI_ID = u.URI_ID and u.URI_STRING = ?"));
                statement.setString(1, uri.toString());
                statement.executeUpdate();
            } finally {
                close(statement);
            }
        } else {
            //otherwise remove all related to the uri
            clearBinding(connection, uri);
        }

    }


}