/*
 * $Header: /home/cvspublic/jakarta-slide/src/webdav/server/org/apache/slide/webdav/util/RedirectRefWorkaroundWithProvidedProperties.java,v 1.2 2005/01/10 18:12:39 luetzkendorf Exp $
 * $Revision: 1.2 $
 * $Date: 2005/01/10 18:12:39 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.webdav.util;

import org.apache.slide.common.PropertyName;
import org.apache.slide.common.SlideException;
import org.apache.slide.content.NodeProperty;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.search.PropertyProvider;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * This is an adapter that creates a RequestedResource from a given
 * NodeRevisionDescriptor(s) for cases when the requesting user-agent does not
 * support redirect references.
 *
 * @author <a href="mailto:snmvaughan92@yahoo.com>snmvaughan92</a>
 */
public class RedirectRefWorkaroundWithProvidedProperties
   extends ResourceWithProvidedProperties {

   /**
    * Create a RequestedResource extends ResourceWithProvidedProperties to
    * present a "text/url" file resource.
    * 
    * @param      revisionDescriptors  the NodeRevisionDescriptors this
    *                                  resource is based on.
    * @param      revisionDescriptor   the NodeRevisionDescriptor this
    *                                  resource is based on.
    * @param      propertyProvider     the PropertyProvider that provides the
    *                                  additional properties (may be
    *                                  <code>null</code>).
    */
   public RedirectRefWorkaroundWithProvidedProperties(
      NodeRevisionDescriptors revisionDescriptors,
      NodeRevisionDescriptor revisionDescriptor,
      PropertyProvider propertyProvider) {
      super(revisionDescriptors, revisionDescriptor, propertyProvider);
   }


   /**
    * Return the property associated with the specified property name and
    * namespace except:
    * <ul>
    * <li><code>resourcetype</code> is always an empty string in order to make
    *     the resource appear to be a file resource.  WebFolders treat all
    *     non-file resource types as collections.</li>
    * </ul>
    * 
    * @see ResourceWithProvidedProperties#getProperty(java.lang.String,
    *         java.lang.String)
    */
   public NodeProperty getProperty(String name, String namespace)
      throws SlideException {
      if ("DAV:".equals(namespace)) {
         if ("resourcetype".equals(name)) {
            return new NodeProperty(name, "", namespace);
         }
      }

      return super.getProperty(name, namespace);
   }


   /**
    * Instead of directly getting the properties from the revision descriptor,
    * we use the list of all property names and call <code>getProperty</code>.
    * This allows us to have a single method which contains all workaround
    * code.
    * 
    * @see ResourceWithProvidedProperties#getAllProperties()
    */
   public Iterator getAllProperties() throws SlideException {
      Set properties = new HashSet();

      Iterator names = getAllPropertiesNames();
      while (names.hasNext()) {
         PropertyName name = (PropertyName) names.next();
         properties.add(getProperty(name));
      }

      return properties.iterator();
   }
}
