/*
 * $Header: /home/cvspublic/jakarta-slide/src/webdav/server/org/apache/slide/webdav/util/properties/CheckoutSetProperty.java,v 1.4 2005/01/10 18:09:22 luetzkendorf Exp $
 * $Revision: 1.4 $
 * $Date: 2005/01/10 18:09:22 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.webdav.util.properties;

import java.util.Iterator;

import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.PropertyName;
import org.apache.slide.common.SlideException;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.content.NodeProperty.NamespaceCache;
import org.apache.slide.search.RequestedResource;
import org.apache.slide.search.Search;
import org.apache.slide.search.SearchQuery;
import org.apache.slide.search.SearchQueryResult;
import org.apache.slide.util.XMLValue;
import org.apache.slide.webdav.util.DaslConstants;
import org.apache.slide.webdav.util.DeltavConstants;
import org.apache.slide.webdav.util.PropertyHelper;
import org.apache.slide.webdav.util.VersioningHelper;
import org.apache.slide.webdav.util.WebdavContext;
import org.apache.slide.webdav.util.WebdavUtils;
import org.apache.slide.webdav.util.resourcekind.ResourceKind;
import org.apache.slide.webdav.util.resourcekind.Version;
import org.jdom.Element;


/**
 * Computes the DeltaV <code>checkout-set</code> property.
 * 
 * <p>Returns an XMLValue containing <code>&lt;href&gt;</code> elements
 * with the URI of the VCRs that have a <code>&lt;checked-out&gt;</code>
 * property pointing to this VR.
 * If the resource is not a VR, the returned XMLValue is empty.
 */
public class CheckoutSetProperty extends AbstractComputedProperty
{
    private Element checkoutSetQueryElement = null;
    private Element checkoutSetQueryLiteralElement = null;
    
    public PropertyName getPropertyName() {
        return DeltavConstants.PN_CHECKOUT_SET;
    }

    public Object computeValue(
            NamespaceAccessToken nsaToken,
            NodeRevisionDescriptors revisionDescriptors, 
            NodeRevisionDescriptor revisionDescriptor, 
            ResourceKind resourceKind, WebdavContext context)
        throws SlideException
    {
        XMLValue xmlValue = new XMLValue();
        
        if (resourceKind instanceof Version) {
            
            String resourcePath = VersioningHelper.getUri(
                    nsaToken,
                    context.getSlideToken(),
                    nsaToken.getContentHelper(),
                    revisionDescriptors,
                    revisionDescriptor);
            
            Element basicSearch = getCheckoutSetQueryElement(nsaToken, resourcePath);
            
            String grammarNamespace = basicSearch.getNamespaceURI();
            Search searchHelper = nsaToken.getSearchHelper();

            SearchQuery searchQuery = searchHelper.createSearchQuery(
                    grammarNamespace,
                    basicSearch,
                    context.getSlideToken(),
                    Integer.MAX_VALUE,
                    context.getContextPath());
            
            SearchQueryResult queryResult = searchHelper.search(
                    context.getSlideToken(), searchQuery);
            Iterator queryResultIterator = queryResult.iterator();
            RequestedResource requestedResource = null;
            Element href = null;
            while (queryResultIterator.hasNext()) {
                requestedResource = (RequestedResource)queryResultIterator.next();
                href = new Element(DeltavConstants.E_HREF, DNSP);
                
                href.setText (WebdavUtils.getAbsolutePath (
                                  requestedResource.getUri(), 
                                  context));
                
                xmlValue.add(href);
            }
        }
        
        return xmlValue;
    }

    /**
     * Returns the query document used to search all resources that have
     * a &lt;checked-out&gt; property with a &lt;href&gt; value containing
     * the URI that identifies the given resource.
     *
     * @param      resourcePath  the Uri to search for.
     *
     * @return     the query document.
     */
    protected Element getCheckoutSetQueryElement(NamespaceAccessToken nsaToken,
            String resourcePath) throws SlideException {
        
        if (checkoutSetQueryElement == null) {
            
            checkoutSetQueryElement = new Element(DaslConstants.E_BASICSEARCH, DNSP);
            
            Element select = new Element(DaslConstants.E_SELECT, DNSP);
            checkoutSetQueryElement.addContent(select);
            Element prop = new Element(DeltavConstants.E_PROP, DNSP);
            select.addContent(prop);
            Element checkedOut = new Element(DeltavConstants.P_CHECKED_OUT, DNSP);
            prop.addContent(checkedOut);
            
            Element from = new Element(DaslConstants.E_FROM, DNSP);
            checkoutSetQueryElement.addContent(from);
            Element scope = new Element(DaslConstants.E_SCOPE, DNSP);
            from.addContent(scope);
            Element href = new Element(DeltavConstants.E_HREF, DNSP);
            scope.addContent(href);
            href.setText("");
            Iterator excludeIterator = PropertyHelper.getNonVcrPathExcludeList(nsaToken).iterator();
            while (excludeIterator.hasNext()) {
                scope.addContent((Element)excludeIterator.next());
            }
            
            Element where = new Element(DaslConstants.E_WHERE, DNSP);
            checkoutSetQueryElement.addContent(where);
            Element propcontains = new Element(DaslConstants.E_PROPCONTAINS, NamespaceCache.SLIDE_NAMESPACE);
            where.addContent(propcontains);
            propcontains.addContent((Element)prop.clone());
            checkoutSetQueryLiteralElement = new Element(DaslConstants.E_LITERAL, DNSP);
            propcontains.addContent(checkoutSetQueryLiteralElement);
        }
        checkoutSetQueryLiteralElement.setText(resourcePath);
        
        return checkoutSetQueryElement;
    }
}
