/*
 * $Header: /home/cvspublic/jakarta-slide/src/webdav/server/org/apache/slide/webdav/util/properties/InheritedProperty.java,v 1.3 2005/01/21 16:15:50 luetzkendorf Exp $
 * $Revision: 1.3 $
 * $Date: 2005/01/21 16:15:50 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.webdav.util.properties;

import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.PropertyName;
import org.apache.slide.common.SlideException;
import org.apache.slide.common.SlideToken;
import org.apache.slide.content.Content;
import org.apache.slide.content.NodeProperty;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.structure.ObjectNode;
import org.apache.slide.structure.Structure;
import org.apache.slide.util.conf.Configuration;
import org.apache.slide.util.conf.ConfigurationException;
import org.apache.slide.webdav.util.WebdavContext;
import org.apache.slide.webdav.util.resourcekind.ResourceKind;

/**
 * Property computer for properties that shall be inherited by parent or
 * ancestor resources.
 * 
 * <p>Sample declaration in Domain.xml.
 * <pre>
 * &lt;event classname="org.apache.slide.event.DomainEvent" enable="true"/> 
 * &lt;listener classname="org.apache.slide.webdav.util.resourcekind.ResourceKindConfigurator">
 *   &lt;configuration>
 *     &lt;resource-kind name="DeltavCompliant">
 *       &lt;live-properties>
 *         &lt;live-property name="myProperty" namespace="http://my.company.com/slide"
 *           computer="org.apache.slide.webdav.util.properties.InheritedProperty">
 *           &lt;computer-configuration>
 *             &lt;default-value>a Value to use if no value could be inherited&lt;/default-value>
 *           &lt;/computer-configuration>
 *         &lt;/live-property>
 *       &lt;/live-properties>
 *     &lt;/resource-kind>
 *   &lt;/configuration>
 * &lt;/listener>
 * </pre>
 * 
 * <p><em>Note:</em> This is a sample of a property that may be <em>computed</em>
 * but not <em>protected</em>.
 */
public class InheritedProperty extends AbstractComputedProperty implements
        ConfigurablePropertyComputer
{
    private PropertyName propertyName;
    private PropertyName inheritedPropertyName;
    private String defaultValue = null;

    public void configure(Configuration configuration)
            throws ConfigurationException
    {
        try {
            Configuration c = configuration.getConfiguration("default-value");
            this.defaultValue = c.getValue("");
        } catch (ConfigurationException e) {
            // ignore; no default-value given; is ok too
        }
        
        String name = propertyName.getName();
        try {
            name = configuration.getConfiguration("property-name").getValue();
        } catch (ConfigurationException e) {
            // ignore; no default-value given; is ok too
        }
        String namespace = propertyName.getNamespace();
        try {
            namespace = configuration.getConfiguration("property-namespace").getValue();
        } catch (ConfigurationException e) {
            // ignore; no default-value given; is ok too
        }
        this.inheritedPropertyName = PropertyName.getPropertyName(name, namespace);
    }

    public PropertyName getPropertyName()
    {
        return this.propertyName;
    }

    public void setPropertyName(PropertyName propertyName)
    {
        this.propertyName = propertyName;
    }

    public Object computeValue(NamespaceAccessToken nsaToken,
            NodeRevisionDescriptors revisionDescriptors,
            NodeRevisionDescriptor revisionDescriptor,
            ResourceKind resourceKind, WebdavContext context)
            throws SlideException
    {
        NodeProperty property = revisionDescriptor.getProperty(this.inheritedPropertyName);

        if (property == null) {
            Structure structure = nsaToken.getStructureHelper();
            Content content = nsaToken.getContentHelper();
            SlideToken slideToken = context.getSlideToken();
            
            String uri = revisionDescriptors.getUri();
            ObjectNode node = structure.retrieve(slideToken, uri); 
            while (property == null) {
                node = structure.getParent(slideToken, node);
                if (node == null) {
                    return this.defaultValue;
                }
                revisionDescriptors = content.retrieve(slideToken, node.getUri());
                revisionDescriptor = content.retrieve(slideToken, revisionDescriptors);
                property = revisionDescriptor.getProperty(this.inheritedPropertyName);
            }
        }

        return property.getValue();
    }

}