/*
 * $Header: /home/cvspublic/jakarta-slide/src/webdav/server/org/apache/slide/webdav/util/properties/SupportedPrivilegeSetProperty.java,v 1.3 2005/01/03 15:04:44 luetzkendorf Exp $
 * $Revision: 1.3 $
 * $Date: 2005/01/03 15:04:44 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.webdav.util.properties;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.PropertyName;
import org.apache.slide.common.SlideException;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.security.SecurityImpl;
import org.apache.slide.structure.ActionNode;
import org.apache.slide.util.XMLValue;
import org.apache.slide.webdav.util.AclConstants;
import org.apache.slide.webdav.util.WebdavContext;
import org.apache.slide.webdav.util.resourcekind.ResourceKind;
import org.jdom.Element;
import org.jdom.Namespace;


/**
 * Computes the ACL <code>supported-privilege-set</code> property.
 */
public class SupportedPrivilegeSetProperty extends
        AbstractComputedProperty
{
    public PropertyName getPropertyName() {
        return AclConstants.PN_SUPPORTED_PRIVILEGE_SET;
    }
    
    public Object computeValue(NamespaceAccessToken nsaToken,
            NodeRevisionDescriptors revisionDescriptors,
            NodeRevisionDescriptor revisionDescriptor, ResourceKind resourceKind, WebdavContext context)
            throws SlideException
    {
        Map actionAggregation = ((SecurityImpl)nsaToken.getSecurityHelper())
                .getActionAggregation(context.getSlideToken());
        Set rootSet = new HashSet(actionAggregation.keySet());
        Iterator actions = actionAggregation.keySet().iterator();
        while (actions.hasNext()) {
            ActionNode a = (ActionNode)actions.next();
            Iterator aggregates = ((Set)actionAggregation.get(a)).iterator();
            while (aggregates.hasNext()) {
                ActionNode c = (ActionNode)aggregates.next();
                rootSet.remove(c);
            }
        }
        Element rootSp = new Element(AclConstants.E_SUPPORTED_PRIVILEGE, DNSP);
        Element p = new Element(AclConstants.E_PRIVILEGE, DNSP);
        p.addContent(new Element(AclConstants.E_ALL, DNSP));
        rootSp.addContent(p);
        Iterator roots = rootSet.iterator();
        addElementsForAggregatedActions(rootSp, roots, actionAggregation);
        return new XMLValue(rootSp);
    }

    
    /**
     * Build the tree of an action with all its aggregated actions for use in
     * the result of <code>supported-privilege-set</code> queries.
     * 
     * This method modifies <code>parentActionElement</code>.
     * 
     * @param parentActionElement The action element to which to add the direct
     *                            aggregated acttions to.
     * @param aggregatedActions The direct aggregated actions.
     * @param actionAggregation A map from an action to its direct aggregated
     *                          actions.
     */
    private void addElementsForAggregatedActions(Element parentActionElement, 
            Iterator aggregatedActions, Map actionAggregation) {
        while (aggregatedActions.hasNext()) {
            ActionNode a = (ActionNode)aggregatedActions.next();
            Element sp = new Element(AclConstants.E_SUPPORTED_PRIVILEGE, DNSP);
            Element p = new Element(AclConstants.E_PRIVILEGE, DNSP);
            Namespace actionNamespace = a.getNamespace();
            if (actionNamespace == null) {
                actionNamespace = DNSP;
            }
            p.addContent(new Element(a.getPath().lastSegment(), actionNamespace));
            addElementsForAggregatedActions(p, ((Set)actionAggregation.get(a)).iterator(), actionAggregation);
            sp.addContent(p);
            parentActionElement.addContent(sp);
        }
    }
}
