/*
 * $Header: /home/cvspublic/jakarta-slide/src/webdav/server/org/apache/slide/webdav/util/properties/WorkspaceCheckoutSetProperty.java,v 1.4 2005/01/10 18:09:22 luetzkendorf Exp $
 * $Revision: 1.4 $
 * $Date: 2005/01/10 18:09:22 $
 *
 * ====================================================================
 *
 * Copyright 1999-2002 The Apache Software Foundation 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.slide.webdav.util.properties;

import java.util.Iterator;

import org.apache.slide.common.NamespaceAccessToken;
import org.apache.slide.common.PropertyName;
import org.apache.slide.common.SlideException;
import org.apache.slide.content.NodeRevisionDescriptor;
import org.apache.slide.content.NodeRevisionDescriptors;
import org.apache.slide.search.RequestedResource;
import org.apache.slide.search.Search;
import org.apache.slide.search.SearchQuery;
import org.apache.slide.search.SearchQueryResult;
import org.apache.slide.util.XMLValue;
import org.apache.slide.webdav.util.DaslConstants;
import org.apache.slide.webdav.util.DeltavConstants;
import org.apache.slide.webdav.util.WebdavContext;
import org.apache.slide.webdav.util.WebdavUtils;
import org.apache.slide.webdav.util.resourcekind.ResourceKind;
import org.apache.slide.webdav.util.resourcekind.Workspace;
import org.jdom.Element;


/**
 * Computes the DeltaV <code>workspace-checkout-set</code> property.
 * 
 * <p>Returns an XMLValue containing <code>&lt;href&gt;</code> elements
 * with the URI of the VCRs in the workspace identified the given
 * NodeRevisionDescriptor(s) that have a <code>&lt;checked-out&gt;</code>.
 * If the resource is not a Workspace, the returned XMLValue is empty.
 */
public class WorkspaceCheckoutSetProperty extends
        AbstractComputedProperty
{
    private Element workspaceCheckoutSetQueryElement = null;
    private Element workspaceCheckoutSetQueryHrefElement = null;
    
    public PropertyName getPropertyName() {
        return DeltavConstants.PN_WORKSPACE_CHECKOUT_SET;
    }

    
    /* 
     * @see org.apache.slide.webdav.util.properties.PropertyDefinition#computeValue(org.apache.slide.common.NamespaceAccessToken, org.apache.slide.content.NodeRevisionDescriptors, org.apache.slide.content.NodeRevisionDescriptor, org.apache.slide.webdav.util.WebdavContext)
     */
    public Object computeValue(NamespaceAccessToken nsaToken,
            NodeRevisionDescriptors revisionDescriptors,
            NodeRevisionDescriptor revisionDescriptor, ResourceKind resourceKind, WebdavContext context)
            throws SlideException
    {
        XMLValue xmlValue = new XMLValue();
        
        if (resourceKind instanceof Workspace) {
            
            Element basicSearch = getWorkspaceCheckoutSetQueryElement(revisionDescriptors.getUri());
            String grammarNamespace = basicSearch.getNamespaceURI();
            Search searchHelper = nsaToken.getSearchHelper();
            SearchQuery searchQuery = searchHelper.createSearchQuery(
                    grammarNamespace,
                    basicSearch,
                    context.getSlideToken(),
                    Integer.MAX_VALUE,
                    context.getContextPath());
            SearchQueryResult queryResult = searchHelper.search(
                    context.getSlideToken(), searchQuery);
            Iterator queryResultIterator = queryResult.iterator();
            RequestedResource requestedResource = null;
            Element hrefElement = null;
            while (queryResultIterator.hasNext()) {
                requestedResource = (RequestedResource)queryResultIterator.next();
                hrefElement = new Element(DeltavConstants.E_HREF, DNSP);
                
                hrefElement.setText(WebdavUtils.getAbsolutePath (
                        requestedResource.getUri(), context));
                
                xmlValue.add(hrefElement);
            }
        }
        
        return xmlValue;
    }

    
    private Element getWorkspaceCheckoutSetQueryElement(String scopePath) {
        
        if (workspaceCheckoutSetQueryElement == null) {
            
            workspaceCheckoutSetQueryElement = new Element(DaslConstants.E_BASICSEARCH, DNSP);
            
            Element select = new Element(DaslConstants.E_SELECT, DNSP);
            workspaceCheckoutSetQueryElement.addContent(select);
            Element prop = new Element(DeltavConstants.E_PROP, DNSP);
            select.addContent(prop);
            Element checkedOut = new Element(DeltavConstants.P_CHECKED_OUT, DNSP);
            prop.addContent(checkedOut);
            
            Element from = new Element(DaslConstants.E_FROM, DNSP);
            workspaceCheckoutSetQueryElement.addContent(from);
            Element scope = new Element(DaslConstants.E_SCOPE, DNSP);
            from.addContent(scope);
            workspaceCheckoutSetQueryHrefElement = new Element(
                    DeltavConstants.E_HREF, DNSP);
            scope.addContent(workspaceCheckoutSetQueryHrefElement);
            
            Element where = new Element(DaslConstants.E_WHERE, DNSP);
            workspaceCheckoutSetQueryElement.addContent(where);
            Element isdefined = new Element(DaslConstants.E_ISDEFINED, DNSP);
            where.addContent(isdefined);
            isdefined.addContent((Element)prop.clone());
        }
        workspaceCheckoutSetQueryHrefElement.setText(truncateLeadingSlash(scopePath));
        
        return workspaceCheckoutSetQueryElement;
    }
}
