/*
 * $Header: /home/cvspublic/jakarta-slide/wck/src/org/apache/slide/simple/store/WebdavStoreBulkPropertyExtension.java,v 1.1 2004/12/09 12:17:09 ozeigermann Exp $
 * $Revision: 1.1 $
 * $Date: 2004/12/09 12:17:09 $
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.simple.store;

import java.util.Map;

import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.lock.ObjectLockedException;
import org.apache.slide.security.AccessDeniedException;
import org.apache.slide.simple.reference.WebdavFileStore;
import org.apache.slide.structure.ObjectNotFoundException;

/**
 * Optional extension to the 
 * {@link org.apache.slide.simple.store.BasicWebdavStore basic store} with
 * bulk property call backs.
 *
 * <p>
 * It can be fed by the same adapter as the
 * {@link org.apache.slide.simple.store.WebdavStoreAdapter adapter}!
 * </p>
 * 
 * <p>
 * Be sure to read the Javadocs of the
 * {@link org.apache.slide.simple.store.BasicWebdavStore basic one} first!
 * </p>
 * 
 * <p>
 * Properties will be retrieved by the {@link #getProperties(String)} and stored by  
 * {@link #setProperties(String, Map)} call back in a bulk. In {@link #getProperties(String)} 
 * you will have to return a map where the keys are the names of
 * the property and the values are the values of the properties. The adapter
 * will call {@link #setProperties(String, Map)} passing over all properties
 * whether changed or not.
 * </p
 * 
 * <p>
 * Full qualified names of properties are assembled by the namespace followed by
 * a colon and the local name of the property.
 * </p>
 * 
 * <p>
 * <em>Caution: It is most important to understand that this is no general purpose store. 
 * It has been designed to solely work with access to Slide via WebDAV with general methods.
 * It relies on certain sequences of calls that are done when the Slide core is being accessed through
 * the WebDAV layer. Other sequences are likely to make this store fail.</em>
 * </p>
 * 
 * @see BasicWebdavStore
 * @see WebdavFileStore
 * @see WebdavStoreAdapter
 * @see WebdavStoreSinglePropertyExtension
 * @version $Revision: 1.1 $
 */
public interface WebdavStoreBulkPropertyExtension extends BasicWebdavStore {

    /**
     * Gets all properties associated to the object specified by
     * <code>uri</code>. The returned map has the property names as keys and
     * the values as values. Names are contructed by namespaces and local names
     * together. E.g. property <code>creationdate</code> with namespace
     * <code>DAV</code> would have the name <code>DAV:creationdate</code>.
     * 
     * @param uri
     *            URI of the object, i.e. content resource or folder
     * @return a map from propertiy names to property values, if the name
     *         contains a namespace like in XML it must be prepended to the name
     *         and seperated by a colon; if this store does not support
     *         properties <code>null</code> or an empty map should be returned
     * @throws ServiceAccessException
     *             if any kind of internal error or any unexpected state occurs
     * @throws AccessDeniedException
     *             if the store denies read access to the properties of this
     *             object
     * @throws ObjectNotFoundException
     *             if there is no object at <code>uri</code>
     * @throws ObjectLockedException
     *             if the object has been locked internally
     */
    Map getProperties(String uri) throws ServiceAccessException, AccessDeniedException, ObjectNotFoundException,
            ObjectLockedException;

    /**
     * Stores all properties associated to the object specified by
     * <code>uri</code>. The property map has the property names as keys and
     * the values as values. Names are contructed by namespaces and local names
     * together. E.g. property <code>creationdate</code> with namespace
     * <code>DAV</code> would have the name <code>DAV:creationdate</code>.
     * 
     * @param uri
     *            URI of the object, i.e. content resource or folder
     * @param properties
     *            a map from propertiy names to property values, if the name
     *            contains a namespace like in XML it must be prepended to the
     *            name and seperated by a colon
     * @throws ServiceAccessException
     *             if any kind of internal error or any unexpected state occurs
     * @throws AccessDeniedException
     *             if the store denies write access to the properties of this
     *             object
     * @throws ObjectNotFoundException
     *             if there is no object at <code>uri</code>
     * @throws ObjectLockedException if the object has been locked internally
     */
    void setProperties(String uri, Map properties) throws ServiceAccessException, AccessDeniedException,
            ObjectNotFoundException, ObjectLockedException;
}