/*
 * $Header: /home/cvspublic/jakarta-slide/wck/src/org/apache/slide/simple/store/WebdavStoreMacroAdapter.java,v 1.2 2005/02/09 19:16:28 ozeigermann Exp $
 * $Revision: 1.2 $
 * $Date: 2005/02/09 19:16:28 $
 *
 * ====================================================================
 *
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.slide.simple.store;

import java.security.Principal;
import java.util.Hashtable;

import javax.transaction.xa.Xid;

import org.apache.commons.transaction.util.xa.TransactionalResource;
import org.apache.slide.authenticate.CredentialsToken;
import org.apache.slide.common.Service;
import org.apache.slide.common.ServiceAccessException;
import org.apache.slide.common.ServiceParameterErrorException;
import org.apache.slide.common.ServiceParameterMissingException;
import org.apache.slide.common.Uri;
import org.apache.slide.lock.ObjectLockedException;
import org.apache.slide.security.AccessDeniedException;
import org.apache.slide.simple.reference.WebdavFileStore;
import org.apache.slide.store.MacroStore;
import org.apache.slide.structure.ObjectAlreadyExistsException;
import org.apache.slide.structure.ObjectNotFoundException;
import org.apache.slide.util.logger.Logger;

/**
 * Driving adapter for call back interfaces {@link BasicWebdavStore},
 * extensions and macro call backs.
 * 
 * @see BasicWebdavStore
 * @see WebdavStoreLockExtension
 * @see WebdavStoreBulkPropertyExtension
 * @see WebdavStoreSinglePropertyExtension
 * @see WebdavFileStore
 * @see WebdavStoreAdapter
 * @version $Revision: 1.2 $
 */
public class WebdavStoreMacroAdapter extends WebdavStoreAdapter implements MacroStore {

    protected boolean isCopySupported, isMoveSupported, isDeleteSupported;

    public void setParameters(Hashtable parameters) throws ServiceParameterErrorException,
            ServiceParameterMissingException {
        super.setParameters(parameters);
        try {
            Object quickCheckObject = storeFactory.newInstance();
            isCopySupported = quickCheckObject instanceof WebdavStoreMacroCopyExtension;
            isMoveSupported = quickCheckObject instanceof WebdavStoreMacroMoveExtension;
            isDeleteSupported = quickCheckObject instanceof WebdavStoreMacroDeleteExtension;
        } catch (Exception e) {
            getLogger().log("Initialize call back store " + callBackFactoryClassName, e, LOG_CHANNEL, Logger.CRITICAL);
            throw new ServiceParameterErrorException(this, CALLBACK_FACTORY_PARAMETER);
        }
    }

    public boolean isMacroDeleteSupported() {
        return isDeleteSupported;
    }

    public void macroDelete(Uri targetUri) throws ServiceAccessException, ObjectNotFoundException {
        log("macroDelete(" + targetUri + ")");
        ((MacroTransactionId) getCurrentlyActiveTransactionalResource()).macroDelete(targetUri);
    }

    public boolean isMacroCopySupported() {
        return isCopySupported;
    }

    public void macroCopy(Uri sourceUri, Uri targetUri, boolean overwrite, boolean recursive)
            throws ServiceAccessException, ObjectNotFoundException, ObjectAlreadyExistsException {
        log("macroCopy(" + sourceUri + " -> " + targetUri + ")");
        ((MacroTransactionId) getCurrentlyActiveTransactionalResource()).macroCopy(sourceUri,
                targetUri, overwrite, recursive);
    }

    public boolean isMacroMoveSupported() {
        return isMoveSupported;
    }

    public void macroMove(Uri sourceUri, Uri targetUri, boolean overwrite)
            throws ServiceAccessException, ObjectNotFoundException, ObjectAlreadyExistsException {
        log("macroMove(" + sourceUri + " -> " + targetUri + ")");
        ((MacroTransactionId) getCurrentlyActiveTransactionalResource()).macroMove(sourceUri,
                targetUri, overwrite);
    }

    protected TransactionalResource createTransactionResource(Xid xid) throws ServiceAccessException {
        CredentialsToken token = (CredentialsToken) WebdavStoreAdapter.credentials.get();
        Principal principal = null;
        if (token != null)
            principal = token.getPrincipal();
        return new MacroTransactionId(xid, this, principal, storeFactory, parameters);
    }

    protected TransactionId createTransactionResource(Uri uri) throws ServiceAccessException {
        return new MacroTransactionId(null, this, uri.getToken().getCredentialsToken().getPrincipal(), storeFactory,
                parameters);
    }


    protected static class MacroTransactionId extends WebdavStoreAdapter.TransactionId {
        protected WebdavStoreMacroCopyExtension copyStore;

        protected WebdavStoreMacroDeleteExtension deleteStore;

        protected WebdavStoreMacroMoveExtension moveStore;

        MacroTransactionId(Xid xid, Service service, Principal principal, BasicWebdavStoreFactory storeFactory, Hashtable parameters)
                throws ServiceAccessException {
            super(xid, service, principal, storeFactory, parameters);
            try {
                if (store instanceof WebdavStoreMacroCopyExtension) {
                    copyStore = (WebdavStoreMacroCopyExtension) store;
                }
                if (store instanceof WebdavStoreMacroMoveExtension) {
                    moveStore = (WebdavStoreMacroMoveExtension) store;
                }
                if (store instanceof WebdavStoreMacroDeleteExtension) {
                    deleteStore = (WebdavStoreMacroDeleteExtension) store;
                }
            } catch (Exception e) {
                throw new ServiceAccessException(service, e);
            }
        }

        public void macroDelete(Uri targetUri) throws ServiceAccessException, ObjectNotFoundException {
            if (deleteStore != null) {
                try {
                    deleteStore.macroDelete(targetUri.toString());
                } catch (AccessDeniedException e) {
                    throw new ServiceAccessException(service, e);
                } catch (ObjectLockedException e) {
                    throw new ServiceAccessException(service, e);
                }
            }
        }

        public void macroCopy(Uri sourceUri, Uri targetUri, boolean overwrite, boolean recursive)
                throws ServiceAccessException, ObjectNotFoundException,
                ObjectAlreadyExistsException {
            if (copyStore != null) {
                try {
                    copyStore.macroCopy(sourceUri.toString(), targetUri.toString(), overwrite,
                            recursive);
                } catch (AccessDeniedException e) {
                    throw new ServiceAccessException(service, e);
                } catch (ObjectLockedException e) {
                    throw new ServiceAccessException(service, e);
                }
            }
        }

        public void macroMove(Uri sourceUri, Uri targetUri, boolean overwrite)
                throws ServiceAccessException, ObjectNotFoundException,
                ObjectAlreadyExistsException {
            if (moveStore != null) {
                try {
                    moveStore.macroMove(sourceUri.toString(), targetUri.toString(), overwrite);
                } catch (AccessDeniedException e) {
                    throw new ServiceAccessException(service, e);
                } catch (ObjectLockedException e) {
                    throw new ServiceAccessException(service, e);
                }
            }
        }
    }
}