package org.esupportail.portal.utils;

import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.services.PersonDirectory;
import org.jasig.portal.utils.SmartCache;

/**
 * Users<br>
 * <br>
 * Cette classe propose des mthodes permettant de faciliter la rcupration<br>
 * d'informations concernant les utilisateurs<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 1.02
 * 
 */
public class Users {

    protected static Log log = LogFactory.getLog(Users.class);
    
    private static SmartCache cache_displayname = new SmartCache(60 * 60);
    private static SmartCache cache_mail = new SmartCache(60 * 60);
    
    /**
     * Constructeur
     */
    private Users() {
    }
    
    /**
	 * Etablit la correspondance entre un login et un nom courant<br>
	 * L'attribut uPortal utilis est le "displayName"<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return Le nom courant de l'utilisateur
	 */
	public static String getUserDisplayName(String userid) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getUserDisplayName()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne "Utilisateur inconnu"
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : userid null ou vide");
			}
	        return "Utilisateur inconnu";
	    }
	    
	    // On rcupre le username dans le cache
	    String username = (String)cache_displayname.get(userid);
	    
	    // Username trouv dans le cache
	    if(username != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : " + userid + " trouv\u00E9 dans le cache :" + username);
			}
	        return username;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return "Utilisateur inconnu";
		}
		
		// Rcupration attribut displayName
	    username = (String)attributes.get("displayName");
		
		// Pas d'attribut displayName
	    if(username == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : " + userid + " ne poss\u00E8de pas d'attribut displayName");
			}
	        return "Utilisateur inconnu";
		}
		
		// Mise en cache
		cache_displayname.put(userid, username);
		
		return username;
	}

	/**
	 * Etablit la correspondance entre un login et un nom courant<br>
	 * L'attribut uPortal utilis est le "displayName"<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return Le nom courant de l'utilisateur (null si l'utilisateur est inexistant)
	 */
	public static String getDisplayName(String userid) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getDisplayName()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne "null"
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : userid null ou vide");
			}
	        return null;
	    }
	    
	    // On rcupre le username dans le cache
	    String username = (String)cache_displayname.get(userid);
	    
	    // Username trouv dans le cache
	    if(username != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : " + userid + " trouv\u00E9 dans le cache :" + username);
			}
	        return username;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return null;
		}
		
		// Rcupration attribut displayName
	    username = (String)attributes.get("displayName");
		
		// Pas d'attribut displayName
	    if(username == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : " + userid + " ne poss\u00E8de pas d'attribut displayName");
			}
	        return null;
		}
		
		// Mise en cache
		cache_displayname.put(userid, username);
		
		return username;
	}
	
	/**
	 * Etablit la correspondance entre une liste de login et les noms courants<br>
	 * L'attribut uPortal utilis est le "displayName"<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userids La liste des login des utilisateurs
	 * @return La liste de correspondance login / nom courant
	 */
	public static Map getUsersDisplayName(Collection userids) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getUsersDisplayName()");
		}
	    
	    HashMap res = new HashMap();
	    
	    Iterator i = userids.iterator();
	    while(i.hasNext()) {
	        String login = (String)i.next();
	        String username = getUserDisplayName(login);
	        res.put(login, username);
	    }
	    return res;
	}

	/**
	 * Etablit la correspondance entre un login et un nom courant<br>
	 * L'attribut uPortal utilis est ici spcifi par l'appelant<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return Le nom courant de l'utilisateur
	 */
	public static String getUserDisplayName(String userid, String attribute) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getUserDisplayName()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne "Utilisateur inconnu"
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : userid null ou vide");
			}
	        return "Utilisateur inconnu";
	    }
	    
	    // On rcupre le username dans le cache
	    String username = (String)cache_displayname.get(userid);
	    
	    // Username trouv dans le cache
	    if(username != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : " + userid + " trouv\u00E9 dans le cache :" + username);
			}
	        return username;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return "Utilisateur inconnu";
		}
		
		// Rcupration attribut spcifi
	    username = (String)attributes.get(attribute);
		
		// Pas d'attribut
	    if(username == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserDisplayName() : " + userid + " ne poss\u00E8de pas d'attribut " + attribute);
			}
	        return "Utilisateur inconnu";
		}
		
		// Mise en cache
		cache_displayname.put(userid, username);
		
		return username;
	}
	
	/**
	 * Etablit la correspondance entre un login et un nom courant<br>
	 * L'attribut uPortal utilis est ici spcifi par l'appelant<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return Le nom courant de l'utilisateur (null si l'utilisateur est inexistant)
	 */
	public static String getDisplayName(String userid, String attribute) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getDisplayName()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne null
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : userid null ou vide");
			}
	        return null;
	    }
	    
	    // On rcupre le username dans le cache
	    String username = (String)cache_displayname.get(userid);
	    
	    // Username trouv dans le cache
	    if(username != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : " + userid + " trouv\u00E9 dans le cache :" + username);
			}
	        return username;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return null;
		}
		
		// Rcupration attribut spcifi
	    username = (String)attributes.get(attribute);
		
		// Pas d'attribut
	    if(username == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getDisplayName() : " + userid + " ne poss\u00E8de pas d'attribut " + attribute);
			}
	        return null;
		}
		
		// Mise en cache
		cache_displayname.put(userid, username);
		
		return username;
	}
	
	/**
	 * Etablit la correspondance entre une liste de login et les noms courants<br>
	 * L'attribut uPortal utilis est ici spcifi par l'appelant<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 24h</b><br>
	 * @param userids La liste des login des utilisateurs
	 * @return La liste de correspondance login / nom courant
	 */
	public static Map getUsersDisplayName(Collection userids, String attribute) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getUsersDisplayName()");
		}
	    
	    HashMap res = new HashMap();
	    
	    Iterator i = userids.iterator();
	    while(i.hasNext()) {
	        String login = (String)i.next();
	        String username = getUserDisplayName(login, attribute);
	        res.put(login, username);
	    }
	    return res;
	}
	
	/**
	 * Etablit la correspondance entre un login et une adresse mail<br>
	 * L'attribut uPortal utilis est le "mail"<br>
	 * <b>Les correspondances identifiant / adresse sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return L'adresse mail de l'utilisateur
	 */
	public static String getUserMail(String userid) {
		if(log.isDebugEnabled()) {
		    log.debug("Users::getUserMail()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne ""
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : userid null ou vide");
			}
	        return "";
	    }
	    
	    // On rcupre le mail dans le cache
	    String mail = (String)cache_mail.get(userid);
	    
	    // Username trouv dans le cache
	    if(mail != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : " + userid + " trouv\u00E9 dans le cache :" + mail);
			}
	        return mail;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return "";
		}
		
		// Rcupration attribut mail
	    mail = (String)attributes.get("mail");
		
		// Pas d'attribut mail
	    if(mail == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : " + userid + " ne poss\u00E8de pas d'attribut mail");
			}
	        return "mail";
		}
		
		// Mise en cache
		cache_mail.put(userid, mail);
		
		return mail;
	}
	
	/**
	 * Etablit la correspondance entre un login et une adresse mail<br>
	 * L'attribut uPortal utilis est ici spcifi par l'appelant<br>
	 * <b>Les correspondances identifiant / adresse sont caches pendant 24h</b><br>
	 * @param userid Le login de l'utilisateur
	 * @return L'adresse mail de l'utilisateur
	 */
	public static String getUserMail(String userid, String attribute) {
	    if(log.isDebugEnabled()) {
		    log.debug("Users::getUserMail()");
		}
	    
	    // Si le login fourni est nul ou vide, on retourne ""
	    if(userid == null || userid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : userid null ou vide");
			}
	        return "";
	    }
	    
	    // On rcupre le mail dans le cache
	    String mail = (String)cache_mail.get(userid);
	    
	    // Mail trouv dans le cache
	    if(mail != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : " + userid + " trouv\u00E9 dans le cache :" + mail);
			}
	        return mail;
	    }
		
		// Rcupration des attributs de personne
	    Hashtable attributes = PersonDirectory.instance().getUserDirectoryInformation(userid);

	    // Pas d'attributs trouvs pour cette personne
	    if(attributes == null || attributes.isEmpty()) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : aucun utilisateur " + userid + " trouv\u00E9");
			}
	        return "";
		}
		
		// Rcupration attribut spcifi
	    mail = (String)attributes.get(attribute);
		
		// Pas d'attribut
	    if(mail == null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Users::getUserMail() : " + userid + " ne poss\u00E8de pas d'attribut " + attribute);
			}
	        return "";
		}
		
		// Mise en cache
		cache_mail.put(userid, mail);
		
		return mail;
	}
}
