package org.esupportail.portal.utils;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.groups.GroupsException;
import org.jasig.portal.groups.IEntity;
import org.jasig.portal.groups.IEntityGroup;
import org.jasig.portal.groups.IGroupMember;
import org.jasig.portal.services.GroupService;
import org.jasig.portal.utils.SmartCache;

/**
 * Groups<br>
 * <br>
 * Cette classe propose des mthodes permettant de faciliter la rcupration<br>
 * d'informations concernant les groupes<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:mathieu.larchet@univ-nancy2.fr">Mathieu Larchet</a>
 * @version 1.0
 * 
 */
public class Groups {
    
    protected static Log log = LogFactory.getLog(Groups.class);
    
    private static SmartCache cache = new SmartCache(60 * 60);
    
    /**
     * Constructeur
     */
    private Groups() {
    }
    
    /**
	 * Etablit la correspondance entre un identifiant de groupe et son nom<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 1h</b><br>
	 * @param groupid L'identifiant du groupe
	 * @return Le nom du groupe
	 * @throws GroupsException
	 */
	public static String getGroupName(String groupid) throws GroupsException {
	    if(log.isDebugEnabled()) {
		    log.debug("Groups::getGroupName()");
	    }
		
		// Si l'identifiant fourni est null ou vide, on lance une exception
	    if(groupid == null || groupid.equals("")) {
	        if(log.isDebugEnabled()) {
			    log.debug("Groups::getGroupName() : groupid null ou vide");
			}
	        throw new GroupsException("Identifiant null ou vide");
	    }
	    
	    // On rcupre le groupname dans le cache
	    String groupname = (String)cache.get(groupid);
	    
	    // Groupname trouv dans le cache
	    if(groupname != null) {
	        if(log.isDebugEnabled()) {
			    log.debug("Groups::getGroupName() : " + groupid + " trouv\u00E9 dans le cache :" + groupname);
			}
	        return groupname;
	    }
	    
		IEntityGroup myGroup = null;

		// Rcupration du groupe
		myGroup = GroupService.findGroup(groupid);
		
		// Groupe introuvable
		if(myGroup == null) {
		    if(log.isDebugEnabled()) {
			    log.debug("Groups::getGroupName() : Groupe " + groupid + " introuvable");
			}
		    throw new GroupsException("Groupe " + groupid + " introuvable");
		}	
		    
		groupname = myGroup.getName();
		
		// Mise en cache
		cache.put(groupid, groupname);
		
		return groupname;
	}
	
	/**
	 * Etablit la correspondance entre une liste d'identifiants de groupes et leur nom<br>
	 * <b>Les correspondances identifiant / nom sont caches pendant 1h</b><br>
	 * @param groupids La liste des identifiants de groupe
	 * @return La liste de correspondance identifiant / nom du groupe
	 * @throws GroupsException
	 */
	public static Map getGroupsName(Collection groupids) throws GroupsException {
	    if(log.isDebugEnabled()) {
		    log.debug("Groups::getGroupsName()");
		}
	    
	    HashMap res = new HashMap();
	    
	    Iterator i = groupids.iterator();
	    while(i.hasNext()) {
	        String id = (String)i.next();
	        String groupname = getGroupName(id);
	        res.put(id, groupname);
	    }
	    return res;	    
	}

	/**
	 * Indique si un utilisateur est membre d'un groupe
	 * @param userid L'identifiant de l'utilisateur
	 * @param group L'identifiant du groupe
	 * @return true si l'utilisateur est membre, false sinon
	 * @throws GroupsException
	 */
	public static boolean isMemberOf(String userid, String group) throws GroupsException {
	    if(log.isDebugEnabled()) {
		    log.debug("Groups::isMemberOf()");
	    }
		
	    // Rcupration de l'utilisateur
	    IEntity user = null;
		user = GroupService.getEntity(userid, org.jasig.portal.security.IPerson.class);
			
		// Rcupration des groupes
		if(user != null) {
			Iterator i = user.getAllContainingGroups();
			while(i.hasNext()) {
				IGroupMember tmp = (IGroupMember)i.next();
					
				// Test si le groupe courant == groupe recherch
				if(tmp.getKey().equals(group)) {
					return true;
				}
			}
			return false;
		}
		else {
			throw new GroupsException("Aucun utilisateur " + userid + " trouv\u00E9");
		}
	}
	
	/**
	 * Indique si un utilisateur est membre d'un des groupes d'une liste
	 * @param userid L'identifiant de l'utilisateur
	 * @param groups La liste des identifiants de groupe
	 * @return true si l'utilisateur est membre d'au moins un des groupes , false sinon
	 * @throws GroupException
	 */
	public static boolean isMemberOf(String userid, Collection groups) throws GroupsException {
		if(log.isDebugEnabled()) {
		    log.debug("Groups::isMemberOf()");
		}
	    
	    Iterator i = groups.iterator();
		while(i.hasNext()) {
			String group = (String)i.next();
			if(log.isDebugEnabled()) {
			    log.debug("PortalGroupManager::isMemberOf() : " + userid + " " + group);
			}
			if(isMemberOf(userid, group)) {
				return true;
			}
		}
		return false;
	}

}
