package org.esupportail.portal.utils.channels;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Priority;
import org.esupportail.portal.utils.channels.plugins.Message;
import org.esupportail.portal.utils.channels.plugins.MessageBean;
import org.jasig.portal.ChannelCacheKey;
import org.jasig.portal.ChannelRuntimeData;
import org.jasig.portal.ChannelRuntimeProperties;
import org.jasig.portal.ChannelStaticData;
import org.jasig.portal.ICacheable;
import org.jasig.portal.IChannel;
import org.jasig.portal.IMimeResponse;
import org.jasig.portal.IServant;
import org.jasig.portal.PortalEvent;
import org.jasig.portal.PortalException;
import org.jasig.portal.utils.SoftHashMap;
import org.xml.sax.ContentHandler;

/**
 * MainChannel<br>
 * <br>
 * Implmentation "type" d'une classe principale d'un channel (MainChannel) <br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:olivier.ziller@univ-nancy2.fr">Olivier Ziller</a>
 * @version $Revision: 1.1 $ 
 */

public abstract class MainChannel implements IChannel, ICacheable, IMimeResponse {

	protected static Log logguer = LogFactory.getLog(MainChannel.class);
    
    protected IConfigActions configActions = null;
	protected ISubChannel currentSubChannel = null;
	protected IServant currentServant = null;
	protected ChannelRuntimeData runtimeData = null;
	protected ChannelStaticData staticData;
	protected Object[] servantResults;
	private HashMap subChannelsMap = new HashMap();
	protected Action currentAction = null;
	protected Action previousAction = null;
	protected boolean inCache = false;
	protected ChannelCacheKey cacheKey;
	protected static SoftHashMap systemCache=new SoftHashMap();
	protected SoftHashMap instanceCache =new SoftHashMap();
	private String defaultAction = "default";
	
	/**
	 * Positionnement des RuntimeData
	 * Il s'agit du "coeur" du processus qui va soit :
	 * 	- Instancier(ou rutiliser) une sous-channel et lancer son cycle de vie : init/setXML/setOutput
	 *  - Instancier un servant ou passer la main au servant dj instanci
	 * 
	 * @param rd Les runtimeData fourni par le socle uPortal
	 * @throws PortalException
	 */
	public void setRuntimeData(ChannelRuntimeData rd) throws PortalException  {
		
		this.runtimeData = rd;
		
		// logs des paramtres reus
		logParams();

		// Pas en mode cache par dfaut
		inCache = false;
		// Si dj en mode servant
		if (currentAction != null)
			if (currentAction.isServant()) {
				currentServant.setRuntimeData(rd);
				//Le servant n'est pas encore fini, on lui passe la main et c'est tout
				if (!currentServant.isFinished()) {
					return;
				}
				// Le servant est fini
				else {
					// On recherche alors l'action  effectuer
					rd.setParameter("action", currentAction.getServantfinish());
					if(logguer.isDebugEnabled()) {
					    logguer.debug("MainChannel::setRuntimeData() : servant fini passe sur " + currentAction.getServantfinish());
					}
					servantResults = currentServant.getResults();
					currentServant = null;
				}
			}
					
		
		// Recherche de l'action en cours
		calcCurrentAction(rd);
		// Si on en trouve pas
		if (currentAction == null)
			throw new PortalException("Impossible de d\u00E9terminer l'action en cours !");
		
		// Action servant ?
		if (currentAction.isServant()) {
			// Instanciation du servant
			currentServant = ServantFactory.instantiateServant(this, currentAction);
		}		
		// Action normale ?
		else {
			// Recherche de la classe associe  l'action dans notre map des classes SubChannel dj instancies
			currentSubChannel = (ISubChannel)subChannelsMap.get(currentAction.getClassname()); //getName()
			// Si pas trouv, instanciation et enregistrement dans la map pour rutilisation ultrieure
			if (currentSubChannel == null ) {
				currentSubChannel = SubChannelFactory.instantiateAction(this, currentAction);
				subChannelsMap.put(currentAction.getClassname(), currentSubChannel); //getName
			}
			else {
				// Sinon (si dj instancie), alors appel de la mthode clear (par dfaut rinitialise les paramres XSL)
				currentSubChannel.clearChannel();
			}
			// CYCLE DE VIE D'UNE SOUS-CHANNEL
			if (currentSubChannel != null) {
				// Calcul de la cl du cache pour l'action en cours
				setCacheKey();
			}
			// On regarde si on est dj dans le cache
			if (cacheKey != null) {
				if (cacheKey.getKeyScope()==ChannelCacheKey.SYSTEM_KEY_SCOPE) {
					ChannelCacheKey key=(ChannelCacheKey)systemCache.get(cacheKey.getKey());
					 if(key!=null) {
					 	if (isCacheValid(key.getKeyValidity())) {
					 		inCache = true;
					 		if(logguer.isDebugEnabled()) {
					 		    logguer.debug("MainChannel::setRuntimeData() : <<< systemcache >>>");
					 		}
					 		return;
					 	}
					 	else
					 		systemCache.remove(cacheKey.getKey());
					 }
					 systemCache.put(cacheKey.getKey(), cacheKey);
				}
				else {
					ChannelCacheKey key=(ChannelCacheKey)instanceCache.get(cacheKey.getKey());
					 if(key!=null) {
					 	if (isCacheValid(key.getKeyValidity())) {
					 		inCache = true;
					 		if(logguer.isDebugEnabled()) {
					 		    logguer.debug("MainChannel::setRuntimeData() : <<< instancecache >>>");
					 		}
					 		return;
					 	}
					 	else
					 		instanceCache.remove(cacheKey.getKey());
					 } 	
					instanceCache.put(cacheKey.getKey(), cacheKey);
				}
			}
			try {
				// init
				if (SubChannelFactory.callInit(currentSubChannel, currentAction, rd) == Boolean.TRUE)
					// setXML
					if (SubChannelFactory.callSetXML(currentSubChannel, currentAction, rd) == Boolean.TRUE)
						// setOutput
						SubChannelFactory.callSetOutput(currentSubChannel, currentAction, rd);
			} 
			catch (FrameWorkException e) {
			    logguer.error("MainChannel::setRuntimeData() : FrameworkException :\n" + e);
				Message.message(this,rd,new MessageBean(e.getDisplayMessage()));
			}
			catch(Throwable e) {
			    logguer.error("MainChannel::setRuntimeData() : Exception :\n" + e);
			}		
		}
	}
			
	/**
	 * Dtermination de l'action en cours  partir des paramres HTTP
	 * @param rd runtimeData de la channel principale
	 * @throws PortalException
	 */
	public void calcCurrentAction(ChannelRuntimeData rd) throws PortalException {
		String actionName = null;
		
		// Enregistrement de l'action prcdente
		previousAction = currentAction;

		// Recherche paramtre action dans les paramtres HTTP
		actionName = rd.getParameter("action");
		// Si pas d'action prcise
		if (actionName == null) {
			// Si pas encore eu d'action, on passe sur l'action default
			if (currentAction == null) {
				actionName = getDefaultAction();
			}
			else {
				// Sinon on retourne sur l'action en cours
				actionName = currentAction.getName();
			}
		}
		if(logguer.isDebugEnabled()) {
 		    logguer.debug("MainChannel::calcCurrentAction() : action courante = " + actionName);
		}
		if (previousAction != null) {
		    if(logguer.isDebugEnabled()) {
	 		    logguer.debug("MainChannel::calcCurrentAction() : action precedente = " + previousAction.getName());
		    }
		}
		
		// Recherche de la classe lie  l'action
		currentAction = (Action)configActions.getActions().get(actionName);
		
		if (currentAction == null) {
			throw new PortalException("Impossible de trouver l'action " + actionName);
		}	
	}
	
	/**
	 * Rendu XML/XSLT de la channel principale qui consiste en une dlguation du rendu soit 
	 * la sous-channel en cours soit au servant en cours
	 * La MainChannel n'effectue pas de rendu propre.
	 * @param out
	 * @throws PortalException
	 */
	public void renderXML(ContentHandler out) throws PortalException {
		// Si on est en train d'afficher un servant
		if ((currentAction.isServant()) && (currentServant != null)) {
			currentServant.renderXML(out);
		}
		else {
			// Si on est pas en cache
			if (currentSubChannel != null) {
				currentSubChannel.renderXML(out);
			}
		}
	}
	
	/**
	 * Effectue une "redirection" depuis une action vers une autre
	 * Attention : cette redirection ne peut-tre faite que dans les mthode init, setXML ou setOutput!!!
	 * @param rd runtimeData  utiliser pour la nouvelle action
	 * @param action action vers laquelle se rediriger
	 * @throws PortalException
	 */
	public void redirect(ChannelRuntimeData rd, String action) throws PortalException {
		rd.setParameter("action", action);
		setRuntimeData(rd);
	}
	
	/**
	 * Initialisation du servant
	 * Cette mthode est appelle par ServantFactory juste aprs l'instanciation du servant.
	 * Elle permet par exemple de positionner des paramtres dans la staticData (dans ce cas,
	 * surclasser la mthode initServant mais ne pas oublier de faire un super.initServant(servant)
	 *  la fin)
	 * @param servant le servant  initialiser
	 * @throws PortalException
	 */
	public void initServant(IServant servant) throws PortalException {
		if (servant != null) {
			servant.setStaticData(staticData);
			// On recherche l'action  initialiser pour ce servant
			String servantInit = getCurrentAction().getServantinit();
			if (servantInit != null) {
				runtimeData.setParameter("action", servantInit);
			}
			else {
				// Si pas d'action prcise : default
				runtimeData.setParameter("action", getDefaultAction());
			}
			servant.setRuntimeData(runtimeData);
		}
	}
	
	/**
	 * Permet de tester si l'action en cours est celle dont le nom est pass en paramtre
	 * @param actionName nom de l'action  comparer  celui de l'action en cours
	 * @return
	 */
	public boolean isCurrentAction(String actionName) {
		if (currentAction == null) {
			return false;
		}
		else {
			return (currentAction.getName().equalsIgnoreCase(actionName));
		}
	}
	
	/**
	 * Retourne une rfrence vers la configuration des actions
	 * @return
	 */
	public IConfigActions getConfigActions() {
		return configActions;
	}

	/**
	 * Positionne la rfrence vers la configuration des actions
	 * @param actions
	 * @throws PortalException
	 */
	public void setConfigActions(IConfigActions actions) throws PortalException {
		try {
			configActions = (IConfigActions)actions.clone();
			// enregistrement plugin message pour gestion des FrameworkException
			Message.register(this);
		}
		catch (CloneNotSupportedException e) {
			throw new PortalException("Impossible de cloner actions");
		}
		logConfigActions();
	}

	/**
	 * Action par dfaut : "default"<br>
	 * Surcharger cette mthode permet de redfinir<br>
	 * l'action par dfaut
	 * @return le nom de l'action par dfaut
	 */
	public String getDefaultAction() {
	    return defaultAction;
	}
	
	/**
	 * Action par dfaut : "default"<br>
	 * Surcharger cette mthode permet de redfinir<br>
	 * l'action par dfaut
	 * @return le nom de l'action par dfaut
	 */
	public void setDefaultAction(String defaultAction) {
	    this.defaultAction = defaultAction;
	}
	
	/**
	 * Action en cours
	 * @return le nom de l'action en cours
	 */
	public Action getCurrentAction() {
		return currentAction;
	}

	/**
	 * Action prcdente
	 * @return le nom de l'action prcdente
	 */
	public Action getPreviousAction() {
		return previousAction;
	}

	/**
	 * Donne le rsultat du servant
	 * @return retourne le rsultat du servant
	 */
	public Object[] getServantResults() {
		return servantResults;
	}
	
	/**
	 * Loggue les paramtres runtimeData reus par la MainChannel
	 * A retrouver dans le log principale uPortal
	 */
	public void logParams() {
		// Rcupration des paramtres de l'application
		if(logguer.isDebugEnabled()) {
		    logguer.debug("MainChannel::logParams() : <params>");
		    Enumeration e = runtimeData.getParameterNames();
		    while (e.hasMoreElements()) {
		        Object obj = e.nextElement();
		        try {
		            String value = runtimeData.getParameter((String)obj);
		            logguer.debug("MainChannel::logParams() : " + obj.toString() + " = " + value);
		        }
		        catch(ClassCastException c) {
		            logguer.debug("MainChannel::logParams() : ClassCastException :" + obj.toString());
		        }
		    }
		    logguer.debug("MainChannel::logParams() : </params>");
		}
	}
	
	/**
	 * Loggue les actions de la channel dans le log uPortal
	 */
	public void logConfigActions() {
		if(logguer.isDebugEnabled()) {
		    logguer.debug("MainChannel::logConfigActions() : log="+ getConfigActions().getLog());
		    logguer.debug("MainChannel::logConfigActions() : <ACTIONS>");
		    Iterator iter = configActions.getActions().keySet().iterator();
		    while (iter.hasNext()) {
		        Action action = (Action)configActions.getActions().get((String)iter.next());
		        logguer.debug("MainChannel::logConfigActions() : ->" + action.getName());
		        logguer.debug("MainChannel::logConfigActions() : 	rendertype=" + action.getRenderType());
		        logguer.debug("MainChannel::logConfigActions() : 	classname=" + action.getClassname());
		        logguer.debug("MainChannel::logConfigActions() : 	xslfile=" + action.getXslFile());
		        logguer.debug("MainChannel::logConfigActions() : 	sslfile=" + action.getSslFile());
		        logguer.debug("MainChannel::logConfigActions() : 	ssltitle=" + action.getSslTitle());
		        logguer.debug("MainChannel::logConfigActions() : 	cachetype=" + action.getCachetype());
		    }
		    logguer.debug("MainChannel::logConfigActions() : </ACTIONS>");
		}
	}

	/**
	 * Effacement de la liste des subChannels dj instancies
	 */
	public void clearChannelsMap() {
		subChannelsMap.clear();
	}
	
	/**
	* Retourne l'identifiant  utiliser dans les pages html pour les noms d'objets
	* @return un identifiant unique qui correspond  la chane "pref" + id de la channel 
	*/
	public String getPrefForm() {
		// on prfixe avec pref pour tre sr que l'id ne commence pas par un chiffre et que a ne fasse pas planter le javascript
		String res = "pref"+staticData.getChannelSubscribeId();
		// pour viter les erreurs javascript lorsque l'utilisation des fragments donne des channelSubscribeId avec des "-"
		res = res.replaceAll("-","_");
		return res;
	}
	
	/**
	 * Permet de rendre unique un nom de paramtre en le prfixant par l'identifiant unique de la classe
	 * @param param nom du paramtre
	 * @return
	 */
	public String getPrefParam(String param) {
		return getPrefForm() + param;
	}
	
	/**
	 * Mise  null du rsultat d'un servant
	 */
	public void clearServantResults() {
		servantResults = null;
	}
	
	/**
	 * Retourne les staticData de la channel principale
	 * @return 
	 */
	public ChannelStaticData getStaticData() {
		return staticData;
	}

	/** 
	 * Receive static channel data from the portal
	 * @param sd static channel data
	 */
	public void setStaticData(ChannelStaticData sd) throws PortalException {
		staticData = sd;		
	}
	
	/** 
	 *  Process layout-level events coming from the portal.
	 *  Satisfies implementation of IChannel Interface.
	 *  @return <b>ChannelRuntimeProperties</b>
	 */
	public ChannelRuntimeProperties getRuntimeProperties() { 
	    return new ChannelRuntimeProperties();
	    }
	 
	/** 
	 *  Process layout-level events coming from the portal.
	 *  Satisfies implementation of IChannel Interface.
	 *  @param ev <b>PortalEvent</b> a portal layout event
	 */
	public void receiveEvent(PortalEvent ev) {
	}
	
	/**
	 * Indique si le cache est actif
	 * @return retourne inCache.
	 */
	public boolean isInCache() {
		return runtimeData.isEmpty();
	}
	
	/**
	 * Implmentation de l'interface ICacheable
	 */
	public ChannelCacheKey generateKey() {
		if (currentAction.isServant()) {
			if (currentServant instanceof ICacheable)
				return ((ICacheable)currentServant).generateKey();
			else
				return null;
		}
		else
			return cacheKey;
	}

	/**
	 * Implmentation de l'interface ICacheable
	 */
	public boolean isCacheValid(Object validity) {
		if (currentAction.isServant()) {
			if (currentServant instanceof ICacheable) {
				return ((ICacheable)currentServant).isCacheValid(validity);
			}
			else {
				return false;
			}
		}
		else {
			return currentSubChannel.isCacheValid(validity);
		}

	}

	/**
	 * Retourne la cl par dfaut du cache pour l'action en cours
	 * @return
	 */
	public ChannelCacheKey getDefaultCacheKey() {
		ChannelCacheKey defaultKey = new ChannelCacheKey();
		
		StringBuffer sbKey = new StringBuffer(1024);
		sbKey.append(currentAction.getName() + ":" +  currentAction.getClassname());
		if (currentAction.getCachetype().equalsIgnoreCase(Action.CHANNEL_CACHE_TYPE)) {
			defaultKey.setKeyScope(ChannelCacheKey.SYSTEM_KEY_SCOPE);
		}
		else {
			defaultKey.setKeyScope(ChannelCacheKey.INSTANCE_KEY_SCOPE);
		}
		defaultKey.setKey(sbKey.toString());
		return defaultKey;
	}
	
	/**
	 * Positionne la cl du cache pour l'action en cours
	 */
	private void setCacheKey() {
		if (currentAction.isServant()) {
			cacheKey = null;
		}
		else {
			cacheKey = currentSubChannel.generateKey();
		}
		if (cacheKey != null) {
		    if(logguer.isDebugEnabled()) {
	 		    logguer.debug("MainChannel::setCacheKey() : cle cache = " + cacheKey.getKey());
		    }
		}
		else {
		    if(logguer.isDebugEnabled()) {
	 		    logguer.debug("MainChannel::setCacheKey() : pas de cle cache");
		    }
		}
	}

	/**
	 * Comportement par dfaut pour indiquer si le cache est encore valide : si 
	 * l'action est cache au niveau channel ou instance alors le cache est toujours valide
	 * sinon cela dpend de isInCache (la channel n'a pas reu de paramtres).
	 * @param validity
	 * @return
	 */
	public boolean getDefaultCacheValid(Object validity) {
		if ((currentAction.getCachetype().equalsIgnoreCase(Action.CHANNEL_CACHE_TYPE)) ||
			(currentAction.getCachetype().equalsIgnoreCase(Action.INSTANCE_CACHE_TYPE))) {
			return true;
		}
		else {
			return isInCache();
		}
	}
	
	/**
	 * Rcupration de la stacktrace dans une string suite  une exception lors de l'invocation dynamaique de mthodes
	 * @param e Exception pour laquelle on veut la stacktrace
	 * @return Stacktrace sous forme de string
	 */
	 static public String stack2string(Exception e) {
		try {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			e.printStackTrace(pw);
			return "------\r\n" + sw.toString() + "------\r\n";
		}
		catch(Exception e2) {
			return "bad stack2string";
		}
	 } 
	 
	/**
	 * Rcupration de la stacktrace dans une string suite  une exception lors de l'invocation dynamaique de mthodes
	 * @param e Exception pour laquelle on veut la stacktrace
	 * @return Stacktrace sous forme d'un flux html
	 */
	 static public String stack2html(Exception e) {
		String res=null;
		try {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			e.printStackTrace(pw);
			res = sw.toString().replaceAll("[\n]","<br>");
			return "------<br>" +  res + "------<br>";
		}
		catch(Exception e2) {
			return "bad stack2string";
		}
	 }
	 
	 /**
	  * Retourne une action de la channel connaissant son nom
	  * @param actionName
	  * @return
	  */
	 public Action getAction(String actionName) {
	 	return (Action)configActions.getActions().get(actionName);
	 }
	 
	 /**
	  * Retourne le paramtre d'une action connaissant le nom de l'action et du paramtre
	  * @param actionName
	  * @param paramName
	  * @return
	  */
	 public ActionParam getActionParam(String actionName, String paramName) {
	 	return (ActionParam)getAction(actionName).getParams().get(paramName);
	 }
	 
	 /**
	  * Envoi d'un message dans le fichier de log pour le niveau de log par dfaut
	  * @param logMessage
	  * @deprecated
	  */
	 public void log(String logMessage) {
	     if(logguer.isDebugEnabled()) {
	         logguer.debug(logMessage);
	     }
	 }
	 
	 /**
	  * Envoi d'un message dans le fichier de log selon le niveau pass en paramtre
	  * @param level
	  * @param logMessage
	  * @deprecated
	  */
	 public void log(Priority level, String logMessage) {
	     if(level.equals(Priority.DEBUG)) {
	         if(logguer.isDebugEnabled()) {
	             logguer.debug(logMessage);
	         }
	     } 
	     if(level.equals(Priority.INFO)) {
	         logguer.info(logMessage);
	     }
	     if(level.equals(Priority.WARN)) {
	         logguer.warn(logMessage);
	     }
	     if(level.equals(Priority.ERROR)) {
	         logguer.error(logMessage);
	     }
	     if(level.equals(Priority.FATAL)) {
	         logguer.fatal(logMessage);
	     }
	 }

	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public void downloadData(OutputStream out) throws IOException {
		((IMimeResponse)currentSubChannel).downloadData(out);
	}

	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public String getContentType() {
		return ((IMimeResponse)currentSubChannel).getContentType();
	}

	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public Map getHeaders() {
		return ((IMimeResponse)currentSubChannel).getHeaders();
	}
	
	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public InputStream getInputStream() throws IOException {
		return ((IMimeResponse)currentSubChannel).getInputStream();
	}

	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public String getName() {
		return ((IMimeResponse)currentSubChannel).getName();
	}

	/**
	 * Implmentation de IMimeResponde
	 * Callback sur l'action en cours
	 */
	public void reportDownloadError(Exception e) {
		((IMimeResponse)currentSubChannel).reportDownloadError(e);
	}
}
