package org.esupportail.portal.utils.channels;

import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Priority;
import org.jasig.portal.ChannelCacheKey;
import org.jasig.portal.ChannelRuntimeData;
import org.jasig.portal.PortalException;
import org.jasig.portal.UPFileSpec;
import org.jasig.portal.utils.XSLT;
import org.xml.sax.ContentHandler;

/**
 * SubChannel<br>
 * <br>
 * Classe abstraite qui propose une implmentation "type" de l'interface ISubChannel <br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:olivier.ziller@univ-nancy2.fr">Olivier Ziller</a>
 * @version $Revision 2.0 $
 * @version $Revision 2.0 $
 */

public abstract class SubChannel implements ISubChannel {

	protected static Log logguer = LogFactory.getLog(SubChannel.class);
    
    protected String INIT_XML;
	protected String START_XML;
	protected String END_XML;

	private String ssl = null;
	private String xsl = null;
	protected Hashtable xslParameters = null; // hashtable de paramtres xsl
	protected String xml = null;
	
	protected MainChannel mainChannel = null; // rfrence vers la channel appellante
	protected ChannelRuntimeData runtimeData = null;
	
	/**
	 * Constructeur d'une SubChannel
	 * Positionne une rfrence vers la channel principale
	 * @param mainChannel instance de la channel principale
	 */
	public SubChannel(MainChannel mainChannel) {
		this.mainChannel = mainChannel;
		xslParameters = new Hashtable();
		// Chaines enveloppe XML
		INIT_XML = "<?xml version=\"1.0\" encoding=\"" +
		  mainChannel.getConfigActions().getXmlEncoding() + "\"?>\n";
		START_XML = INIT_XML + "<xml>\n";
		END_XML = "</xml>\n";
	}
	
	/**
	 * Implmentation ISubChannel
	 */
	public Boolean init(ChannelRuntimeData rd) throws PortalException, FrameWorkException {
		runtimeData = rd;
		return Boolean.TRUE;
	}
	
	/**
	 * Implmentation ISubChannel
	 */
	public void clearChannel() {
		xslParameters.clear();
	}
	
	/**
	 * Implmentation ISubChannel
	 */
	public void setOutput() throws FrameWorkException {
		Action current = mainChannel.getCurrentAction();
		
		if (current.getRenderType().equalsIgnoreCase(Action.RENDER_TYPE_XSL)) {
			setSSL(null);
			setXSL(current.getXslFile());
		}
		else {
			setSSL(current.getSslFile());
			setXSL(current.getSslTitle());
		}
	}

	/**
	 * Implmentation ISubChannel
	 */
	public void renderXML (ContentHandler out) throws PortalException {
		String ssl = getSSL();
		String xsl = getXSL();
		String xml = getXML();
		
		if(logguer.isDebugEnabled()) {
		    logguer.debug("SubChannel::renderXML() : ssl = " + ssl);
		    logguer.debug("SubChannel::renderXML() : xsl = " + xsl);
		    logguer.debug("SubChannel::renderXML() : xml = \n" + xml);
		}
		
		if (xml == null) {
			throw new PortalException("Pas de flux XML disponible");
		}
		if (runtimeData == null) {
		    throw new PortalException("Pas de runtimeData disponibles");
		}			
			
		XSLT xslt = XSLT.getTransformer(this);
		xslt.setXML(xml);
		if (ssl != null) {
			if(logguer.isDebugEnabled()) {
			    logguer.debug("SubChannel::renderXML() : setXML avec ssl");
			}
			xslt.setXSL(ssl, xsl, runtimeData.getBrowserInfo());
		}
		else {
			// utilisation directe d'un xsl
		    if(logguer.isDebugEnabled()) {
			    logguer.debug("SubChannel::renderXML() : setXML sans ssl");
		    }
			xslt.setXSL(xsl);
		}
		addDefaultXslParameters();
		xslt.setStylesheetParameters(getXSLParameter());
		xslt.setTarget(out);
		xslt.transform();
	}

	/**
	 * Implmentation ISubChannel
	 */
	public String getXML() {
		// Si pas de xml encore fournit, renvoie un contenu valide mais vode
		if (xml == null)
			xml = START_XML + END_XML;
		else 
			// Si le xml fourni ne commence pas par la chaine d'initialisation, on imbrique le xml
			// entre START_XML et END_XML
			if (!xml.startsWith(INIT_XML))
				xml = START_XML + xml + END_XML;
		return xml;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public void setXML(String string) throws FrameWorkException {
		xml = string;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public Boolean setXML() throws FrameWorkException {
		return Boolean.TRUE;
	}
	
	/**
	 * Implmentation ISubChannel
	 */
	public Hashtable getXSLParameter() {
		return xslParameters;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public void setXSLParameter(Hashtable hashtable) {
		xslParameters = hashtable;
	}

	/** 
	 * Rcupre l'instance de la channel principale
	 * @return Une rfrence vers l'instance de la channel principale
	 */
	public MainChannel getMainChannel() {
		return mainChannel;
	}

	/**
	 * Ajoute des paramtres XSL par dfaut aux paramtres existant
	 * baseActionURL = url vers la channel en cours
	 * prefForm = prefixe  utiliser dans la feuille xslt pour s'assurer de l'unicit du nom des objects javascript
	 * mediaPath = chemins d'accs aux mdias (images)
	 */
	public void addDefaultXslParameters() {
		xslParameters.put("baseActionURL",runtimeData.getBaseActionURL());
		xslParameters.put("prefForm", getMainChannel().getPrefForm());
		xslParameters.put("mediaPath", getMediaPath());
	}
	
	/**
	 * Ajoute le paramtre pour le download aux paramtres existants
	 * baseDownloadURL = url vers la channel en cours pour une opration de tlchargement
	 */
	public void addDownloadXslParameter() {
		try {
			xslParameters.put("baseDownloadURL",runtimeData.getBaseWorkerURL(UPFileSpec.FILE_DOWNLOAD_WORKER, true).replaceAll("%2F", "/"));
		}
		catch (PortalException e) {
		    logguer.error("MainChannel::addDownloadXSLParameter() : PortalException :\n" + e);
		}
	}

	/**
	 * Efface les paramtres XSL en cours et ajoute ceux par dfaut
	 */
	public void setDefaultXslParameters() { 
	 	xslParameters.clear();
	 	addDefaultXslParameters();
	}

	/**
	 * Rcupre les runtimeData de la sous-channel
	 * @return les runtimeData de la sous-channel
	 */
	public ChannelRuntimeData getRuntimeData() {
		return runtimeData;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public String getSSL() {
		return ssl;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public String getXSL() {
		return xsl;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public void setSSL(String ssl) {
		this.ssl = ssl;
	}

	/**
	 * Implmentation ISubChannel
	 */
	public void setXSL(String xsl) {
		this.xsl = xsl;
	}

	/**
	 * Retourne le chemin d'accs aux mdis 
	 * @return le chemin d'accs aux mdias
	 */
	public String getMediaPath() {
		String mediaPath = getPackageName(this.getClass());
		mediaPath = mediaPath.replaceAll("[.]","/");
		return "media/" + mediaPath + "/";
	}
	
	/**
	 * Retourne le nom du package en cours
	 * @return nom du package
	 */
	public static String getPackageName(Class c) {
		String fullyQualifiedName = c.getName();
	    int lastDot = fullyQualifiedName.lastIndexOf ('.');
	    if (lastDot==-1){ return ""; }
	    
	    return fullyQualifiedName.substring (0, lastDot);
	}

	/**
	 * Implmentation ISubChannel
	 */
	public ChannelCacheKey generateKey() {
		// cl du cache par dfaut
		return getMainChannel().getDefaultCacheKey();
	}
	
	/**
	 * Implmentation ISubChannel
	 */
	public boolean isCacheValid(Object validity) {
		// Validit du cache par dfaut
		return getMainChannel().getDefaultCacheValid(validity);
	}
	
	/**
	 * Implmentation ISubChannel
	 * @deprecated
	 */
	public void log(String logMessage) {
	    if(logguer.isDebugEnabled()) {
	        logguer.debug(logMessage);
	    }
	 }
	
	/**
	 * Implmentation ISubChannel
	 * @deprecated
	 */
	public void log(Priority level, String logMessage) {
	 	if(level.equals(Priority.DEBUG)) {
	 	    if(logguer.isDebugEnabled()) {
	 	        logguer.debug(logMessage);
	 	    }
	 	}
	 	if(level.equals(Priority.INFO)) {
	 	    logguer.info(logMessage);
	 	}
	 	if(level.equals(Priority.WARN)) {
	 	    logguer.warn(logMessage);
	 	}
	 	if(level.equals(Priority.ERROR)) {
	 	    logguer.error(logMessage);
	 	}
	 	if(level.equals(Priority.FATAL)) {
	 	    logguer.fatal(logMessage);
	 	}
	 }
}
