package org.esupportail.portal.utils.channels;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.portal.ChannelRuntimeData;
import org.jasig.portal.IMimeResponse;
import org.jasig.portal.PortalException;
import org.jasig.portal.utils.XSLT;

/**
 * SubChannelAjax<br>
 * <br>
 * Classe abstraite qui propose une implmentation "type" de SubChannel pour une utilisation AJAX <br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="mailto:olivier.ziller@univ-nancy2.fr">Olivier Ziller</a>
 * @version $Revision 2.0 $
 */public abstract class SubChannelAjax extends SubChannel implements IMimeResponse {

	private static final Log log = LogFactory.getLog(SubChannelAjax.class);
	
	// Diffrents type de rponse Ajax
	private static final String AJAX_REPLACE_CONTENT = "replaceContent";
	private static final String AJAX_REDIRECT = "redirect";
	
	/**
	 * Constructeur
	 * 
	 * @param mainChannel
	 */
	public SubChannelAjax(MainChannel mainChannel) {
		super(mainChannel);
	}

	/**
	 * Retourne le type MIME du document
	 * @return Le type MIME
	 */
	public String getContentType() {
		return "text/xml";
	}

	/**
	 * Retourne un InputStream correspondant au fichier  tlcharger
	 * @return Le flux de donnes
	 */
	public InputStream getInputStream() throws IOException {
		return null;
	}

	/**
	 * Mthode qui retourne le xml ajax
	 * @param out Un OutputStream vers le navigateur du client
	 */
	public void downloadData(OutputStream out) throws IOException {
		try {
			OutputStream os = new ByteArrayOutputStream();
			PrintWriter writer = new PrintWriter(out);
			writer.println(renderAjax(os));
			writer.close();
		} catch (Exception e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * Transformations permettant de gnrer le flux XML ajax
	 * 
	 * @param out
	 * @throws PortalException
	 */
	private String renderAjax (OutputStream out) throws PortalException {
		String ssl = getSSL();
		String xsl = getXSL();
		String xml = getXML();
		
		if(logguer.isDebugEnabled()) {
		    logguer.debug("SubChannelAjax::renderAjax() : ssl = " + ssl);
		    logguer.debug("SubChannelAjax::renderAjax() : xsl = " + xsl);
		    logguer.debug("SubChannelAjax::renderAjax() : xml = \n" + xml);
		}
		
		if (xml == null) {
			throw new PortalException("Pas de flux XML disponible");
		}
		if (runtimeData == null) {
		    throw new PortalException("Pas de runtimeData disponibles");
		}			
			
		XSLT xslt = XSLT.getTransformer(this);
		xslt.setXML(xml);
		if (ssl != null) {
			if(logguer.isDebugEnabled()) {
			    logguer.debug("SubChannelAjax::renderXML() : setXML avec ssl");
			}
			xslt.setXSL(ssl, xsl, runtimeData.getBrowserInfo());
		}
		else {
			// utilisation directe d'un xsl
		    if(logguer.isDebugEnabled()) {
			    logguer.debug("SubChannelAjax::renderXML() : setXML sans ssl");
		    }
			xslt.setXSL(xsl);
		}
		addDefaultXslParameters();
		// Pour pouvoir des liens valides
		getXSLParameter().put("baseActionURL",runtimeData.getBaseActionURL(true));
		// Pour signaler qu'on rpond en Ajax
		if (runtimeData.getParameter("ajaxCalled") != null) {
			getXSLParameter().put("ajaxCalled",runtimeData.getParameter("ajaxCalled"));
			runtimeData.remove("ajaxCalled");
		}
		else
			getXSLParameter().put("ajaxCalled","true");
		// id du tag Ajax  gnrer
		String ajaxId = runtimeData.getParameter("ajaxId");
		if (ajaxId != null)
			getXSLParameter().put("ajaxId",ajaxId);
		log.debug("SubChannelAjax::renderXML() : ajaxId=" + ajaxId);
		// id de l'lment HTML  remplacer
		String htmlId = runtimeData.getParameter("htmlId");
		if (htmlId != null)
			getXSLParameter().put("htmlId",htmlId);
		log.debug("SubChannelAjax::renderXML() : htmlId=" + htmlId);
		// Si on a demand une redirection
		if (getRuntimeData().getParameter("AjaxRedirectFullScreen") != null) {
			return generateAjaxRedirectResponse();
		}
		// transformation XSL classique
		xslt.setStylesheetParameters(getXSLParameter());
		logXslParameters();
		xslt.setTarget(out);
		xslt.transform();
		String res = out.toString();
		// Si on ne limite pas le rsultat  un DIV
		if (ajaxId == null) {
			log.debug("SubChannelAjax::renderAjax::ajaxId est null");
			return generateAjaxReplaceContentResponse(res);
		}
		if (ajaxId.equals("")) {
			log.debug("SubChannelAjax::renderAjax::ajaxId est vide");
			return generateAjaxReplaceContentResponse(res);
		}
		// Sinon, restriction du HTML retourn au contenu d'un DIV
		String tag_debut = "<ajax_" + ajaxId + ">";
		String tag_fin = "</ajax_" + ajaxId + ">";
		int debut = res.indexOf(tag_debut);
		int fin = res.indexOf(tag_fin);
		if ((debut < 0) || (fin < 0)) {
			log.info("SubChannelAjax::renderAjax::Impossible de trouver le dbut ou la fin de la balise ajax_" + ajaxId);
			return generateAjaxReplaceContentResponse(res);
		}
		res = res.substring(debut + tag_debut.length(), fin);
		// Si on a demand un retour direct
		if (getRuntimeData().getParameter("directUpdate") != null) {
			log.debug("Ajax-directUpdate="+res);
			return res;
		}
		else
			return generateAjaxReplaceContentResponse(res);
	}
	
	/**
	 * Gnration d'une rponse au format Ajax
	 * 
	 * @param responseType le type de rponse Ajax qui peut-tre REPLACE_CONTENT, REDIRECT
	 * @param response
	 * @return
	 */
	private String generateAjaxResponse(String responseType, String response) {
		String res = "<?xml version=\"1.0\" encoding=\"" +
		  mainChannel.getConfigActions().getXmlEncoding() + "\"?>";
		res = res + "<ajax-response>";
		res = res + "<response type='" + responseType + "'";
		if (responseType == AJAX_REPLACE_CONTENT)
			res = res + " htmlId='" + getRuntimeData().getParameter("htmlId") + "'";
		res = res + ">";
		res = res + "<![CDATA[" + response + "]]>";
		res = res + "</response>";
		res = res + "</ajax-response>";
		log.debug("Ajax-Response="+res);
		return res;
	}

	/**
	 * Gnration d'une rponse Ajax de type REPLACE_CONTENT
	 * 
	 * @param response
	 * @return
	 */
	private String generateAjaxReplaceContentResponse(String response) {
		return generateAjaxResponse(AJAX_REPLACE_CONTENT, response);
	}

	/**
	 * Gnration d'une rponse Ajax de type REDIRECT
	 * 
	 * @param response
	 * @return
	 */
	private String generateAjaxRedirectResponse() {
		String redirect = runtimeData.getBaseActionURL(true) + "?";
		Enumeration e = getRuntimeData().getParameterNames();
		while (e.hasMoreElements()) {
			Object obj = e.nextElement();
			if (obj instanceof String) {
				try {
					Object value =  runtimeData.getParameter((String) obj);
					if (value instanceof String) {
						log.debug("value=" + (String) value);
						redirect = redirect + (String)obj + "=" + (String)value + "&";
					}
				}
				catch (ClassCastException c) {
					log.debug(c);
				}
			}	
		}
		return generateAjaxResponse(AJAX_REDIRECT, redirect);
	}

	/**
	 * Retourne le nom du fichier
	 * @return Le nom du fichier
	 */
	public String getName() {
		return  null;
	}

	/**
	 * Retourne la liste des headers HTTP  envoyer au navigateur
	 * @return La liste des headers
	 */
	public Map getHeaders() {
          return new HashMap();
	}

	/**
	 * Mthode appele si une erreur se produit durant le tlchargement
	 * @param e L'exception leve
	 */
	public void reportDownloadError(Exception e) {
		log.error("SubChannelAjax::reportingDownloadError() : " + e);
	}

	/* (non-Javadoc)
	 * @see org.esupportail.portal.utils.channels.SubChannel#addDefaultXslParameters()
	 */
	public void addDefaultXslParameters() {
		// TODO Auto-generated method stub
		super.addDefaultXslParameters();
		// Ajout du param baseDownloadURL
		addDownloadXslParameter();
	}
	
	/**
	 * @param rd
	 * @param newHtmlId
	 */
	public void replaceHtmlId(ChannelRuntimeData rd, String newHtmlId) {
		rd.setParameter("htmlId", getMainChannel().getPrefForm() + newHtmlId);
		log.debug("SubChannelAjax::replaceHtmlId::newHtmlId=" + rd.getParameter("htmlId"));
	}

	/**
	 * @param rd
	 * @param newHtmlId
	 */
	public void replaceAjaxId(ChannelRuntimeData rd, String newAjaxId) {
		rd.setParameter("ajaxId", newAjaxId);
		log.debug("SubChannelAjax::replaceHtmlId::newAjaxId=" + rd.getParameter("ajaxId"));
	}
}
