package org.esupportail.portal.utils.channels;

import java.io.IOException;
import java.net.URL;
import java.sql.SQLException;
import java.util.HashMap;
import javax.servlet.jsp.JspException;

import org.apache.commons.digester.Digester;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Priority;
import org.esupportail.portal.utils.database.Database;
import org.esupportail.portal.utils.database.IConfigPools;
import org.esupportail.portal.utils.database.Query;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * ConfigChannel<br>
 * <br>
 * Implmentation de rfrence de la classe de lecture de la configuration<br> 
 * <br>
 * (c)Copyright <a href="http://www.univ-nancy2.fr">CRI Nancy 2 2004</a><br>
 * @author <a href="mailto:olivier.ziller@univ-nancy2.fr">Olivier Ziller</a>
 * @version $Revision: 1.7 $
 */

public abstract class ConfigChannel implements IConfigActions, IConfigPools, IConfigMail
{ 
	protected static Log logguer = LogFactory.getLog(ConfigChannel.class);
	/** ClassLoader  utiliser pour charger le fichier de proprits du canal */
	private static ClassLoader loader = null;
    
    private HashMap pools;
	private HashMap actions = null;
	private String xmlencoding;
	private String log;
	private Priority logLevel;
	protected Digester dig = null;
	protected URL resourceURL = null;
	private String packageName;
	private String smtp;
	private String from;
	private String to;
	private String subject;
	private String message;
	private String displayError;
	private HashMap groupmapping;
	
	/**
	* Constructeur
	* @throws IOException
	* @throws JspException
	*/
	protected ConfigChannel() {
		xmlencoding = "ISO-8859-1";
		log = "debug";
		logLevel = Priority.DEBUG;
		packageName = getPackageName();
		initDigester();
		customDigester();
		parseDigester();
	}
	
	/**
	 * Initialisation du digester avec paramtrage pour lecture des actions et des pools 
	 */
	private void initDigester() {
		dig = new Digester();
		dig.push(this);

		// Parsing des actions
		dig.addCallMethod("config/actions","addActions",2);
		dig.addCallParam("config/actions", 0, "xmlencoding");
		dig.addCallParam("config/actions", 1, "log");

			// Une action
			dig.addObjectCreate("config/actions/action",Action.class);
			dig.addSetProperties( "config/actions/action");
		
				// Paramtres de l'action
				dig.addObjectCreate("config/actions/action/param",ActionParam.class);
				dig.addSetProperties( "config/actions/action/param");
				dig.addSetNext("config/actions/action/param","addActionParam");
		
			dig.addSetNext("config/actions/action","addAction");
		
		// Valeurs des paramtres d'action
		dig.addCallMethod("config/actions/action/param/value","addParamValue",1);
		dig.addCallParam("config/actions/action/param/value",0);
		setActions(new HashMap());
		
		// Parsing des pools
		dig.addObjectCreate("config/pools/pool",Database.class);
		dig.addSetProperties("config/pools/pool");
		dig.addSetNext("config/pools/pool","addPool");
		
		// Parsing de la configuration des erreurs
		dig.addCallMethod("config/displayError","addDisplayError",1);
		dig.addCallParam("config/displayError",0);
		
		// Parsing de l'envoi de mail en cas d'erreur
		dig.addCallMethod("config/mailError","addMailError",5);
		dig.addCallParam("config/mailError/smtp",0);
		dig.addCallParam("config/mailError/from",1);
		dig.addCallParam("config/mailError/to",2);
		dig.addCallParam("config/mailError/subject",3);
		dig.addCallParam("config/mailError/message",4);
		
		// Parsing des mappings pour les groupes
		dig.addObjectCreate("config/groupmapping/group",Group.class);
		dig.addSetProperties("config/groupmapping/group");
		dig.addSetNext("config/groupmapping/group","addGroup");
	}
	
	/**
	 * A surcharger pour la partie propre au canal du fichier de configuration
	 */
	protected void customDigester() {
	}
	
	/**
	 * A implmenter pour dfinir o se trouve le fichier de configuration
	 * Exemple : return "/properties/channels/fr_univnancy2/CAdresses/CApogeeAdresses.xml";
	 * @return
	 */
	protected abstract String getConfigFile();

	/**
	 * Parsing du fichier de config par le digester
	 */
	private void parseDigester() {
		if (loader != null)
			resourceURL = loader.getResource(getConfigFile());
		else
			resourceURL = ConfigChannel.class.getResource(getConfigFile());
		if(resourceURL != null) {
		    try {
		        // On parse le fichier dont le chemin est pass en paramtre
				dig.parse(new InputSource(resourceURL.toExternalForm()));
			}
			catch(IOException e) {
				logguer.error("ConfigChannel::ConfigChannel() : Impossible d'ouvrir le fichier de config \n" + e.toString());
			}
			catch(SAXException e) {
				logguer.error("ConfigChannel::ConfigChannel() : SAXException :\n" + e.toString());
			} 
			catch(Exception e) {
				logguer.error("ConfigChannel::ConfigChannel() : Exception :\n" + e.toString());
			}
		}
		else{
			logguer.error("ConfigChannel::ConfigChannel() : Le fichier de configuration est introuvable");
		}	
	}
	
	/**
	 * Mthode appele par digester lorsqu'une action a t lue dans le fichier de config 
	 * @param a
	 */
	public void addAction(Action a) {
		if (a.getClassname().indexOf(".") < 0) {
		    a.setClassname(packageName + "." + a.getClassname());
		}
		actions.put(a.getName(), a);
	}
	
	/**
	 * Retourne la liste des actions
	 * @return
	 */
	public HashMap getActions() {
		return actions;
	}

	/**
	 * Positionne la liste des actions
	 * @param map
	 */
	public void setActions(HashMap map) {
		actions = map;
	}

	/**
	 * Mthode appele par le digester lorsqu'un pool a t lu dans le fichier de config
	 * @param db
	 */
	public void addPool(Database db) {
	    if (pools == null) {
	        pools = new HashMap();
	    }
	    pools.put(db.getName(), db);
	}

	/**
	 * Retourne la liste des pools
	 * @return
	 */
	public HashMap getPools() {
		return pools;
	}
	
	/**
	 * Positionne la liste des pools
	 * @param map
	 */
	public void setPools(HashMap map) {
		this.pools = map;
	}

	/**
	 * Retourne le nom du package en cours
	 * @return
	 */
	protected String getPackageName() {
		String fullyQualifiedName = this.getClass().getName();
	    int lastDot = fullyQualifiedName.lastIndexOf ('.');
	    if(lastDot==-1) {
	        return "";
	    }
	    return fullyQualifiedName.substring (0, lastDot);
	}
	
	/**
	 * Retourne une connexion vers le pool "default"
	 * @return 
	 */
	public Query getConnexionDefault() {
		MAGQuery query = null;
		Database db = null;
		try {
			if(getPools() != null) { 
				db = (Database) getPools().get("default");
			}
			if(db == null) {
				logguer.error("ConfigChannel::getConnexionDefault() : Aucun pool 'default' d\u00E9fini");
			}
			else {
			    query = new MAGQuery(db);
			}
		}
		catch (SQLException e) { 
			logguer.error("ConfigChannel::getConnexionDefault() : SQLException :\n" + e);
		}
		return query;
	}
	
	/**
	 * Retourne une connexion vers le pool spcifi
	 * @return
	 */
	public Query getConnexion(String name) {
		MAGQuery query = null;
		Database db = null;
		try {
			if (getPools() != null) {
				db = (Database)getPools().get(name);
			}
			if(db == null) { 
				logguer.error("ConfigChannel::getConnexion() : Aucun pool '" + name + "' d\u00E9fini");
			}
			else {
			    query = new MAGQuery(db);
			}
		}
		catch (SQLException e) { 
			logguer.error("ConfigChannel::getConnexion() : SQLException :\n" + e);
		}
		return query;
	}
	
	/**
	 * Mthode appele par le digester lorsqu'un affichage d'erreur a t lu dans le fichier de config
	 * @param msg
	 */
	public void addDisplayError(String msg) {
		this.displayError = msg;
	}
	
	/**
	 * Mthode appele par le digester lorsque le mail d'erreur a t lu dans le fichier de config
	 * @param smtp
	 * @param from
	 * @param to
	 * @param subject
	 * @param message
	 */
	public void addMailError(String smtp, String from, String to, String subject, String message){
		this.smtp = smtp;
		this.from = from;
		this.to = to;
		this.subject = subject;
		if(message == null) {
			this.message = "";
		}
		else {	
		    this.message = message;
		}
	}
	
	/**
	 * Retourne le sujet du mail
	 * @return
	 */
	public String getSubject() {
		return subject;
	}
	
	/**
	 * Retourne l'expditeur du mail
	 * @return
	 */
	public String getFrom() {
		return from;
	}
	
	/**
	 * Retourne le destinataire du mail
	 * @return
	 */
	public String getTo() {
		return to;
	}
	
	/**
	 * Retourne le contenu du mail
	 * @return
	 */
	public String getMessage() {
		return message;
	}
	
	/**
	 * Retourne le serveur de mail SMTP
	 * @return
	 */
	public String getSmtp() {
		return smtp;
	}

	/**
	 * Retourne le message d'erreur
	 * @return
	 */
	public String getDisplayError() {
		return displayError;
	}

	/**
	 * Positionne le message d'erreur
	 * @param string
	 */
	public void setDisplayError(String string) {
		displayError = string;
	}

	/**
	 * Retourne l'encodage XML utilis
	 * @return
	 */
	public String getXmlEncoding() {
		return xmlencoding;
	}

	/**
	 * Retourne le niveau de log par dfaut
	 * @return
	 * @deprecated
	 */
	public Priority getLogLevel() {
		return logLevel;
	}
	
	/**
	 * Positionne le niveau de log par dfaut
	 * @param log
	 * @deprecated
	 */
	public void setLog(String log) {
		this.log = log;
		if(log.equalsIgnoreCase("debug")) {
			logLevel = Priority.DEBUG;
		}
		if(log.equalsIgnoreCase("info")) {
			logLevel = Priority.INFO;
		}
		if(log.equalsIgnoreCase("error")) {
			logLevel = Priority.ERROR;
		}
		if (log.equalsIgnoreCase("fatal")) {
			logLevel = Priority.FATAL;
		}
	}
	
	/**
	 * Mthode appele par le digester pour lire les attributs du tag actions
	 * @param xmlencoding
	 * @param log
	 */
	public void addActions(String xmlencoding, String log){
		this.xmlencoding = xmlencoding;
		setLog("INFO");
	}

	/**
	 * Duplique l'objet
	 * @return
	 */
	public Object clone() throws CloneNotSupportedException {
		ConfigChannel copie = (ConfigChannel)super.clone();
		copie.actions = (HashMap)actions.clone();
		return copie;
	}
	
	/**
	 * Retourne le niveau de log par dfaut
	 * @return
	 */
	public String getLog() {
		return log;
	}
	
	/**
	 * Mthode appele par le digester lorsqu'un groupmapping a t lu dans le fichier de config
	 * @param g
	 */
	public void addGroup(Group g) {
		if (groupmapping == null) {
			groupmapping = new HashMap();
		}
		groupmapping.put(g.getLocalname(),g.getUportalid());
	}
	
	/**
	 * Retourne les mappings des groupes
	 * @return
	 */
	public HashMap getGroupmapping() {
		return groupmapping;
	}
	
	/**
	 * Positionne les mappings des groupes
	 */
	public void setGroupmapping(HashMap groupmapping) {
		this.groupmapping = groupmapping;
	}

	/**
	 * Permet de modifier le classLoader (utile pour JUnit)
	 * 
	 * @param loader The loader to set.
	 */
	public static void setLoader(ClassLoader loader) {
		ConfigChannel.loader = loader;
	}
}
