/**
 * Classe utilitaire pour les accs base de donnesClasse utilitaire pour les accs base de donnes<br>
 * <br>
 * (c)Copyright <a href="http://www.esup-portail.org">ESup-Portail 2004</a><br>
 * @author <a href="http://mailto:cedric.champmartin@univ-nancy2.fr">Cdric Champmartin</a>
 * @version $Revision: 1.8 $
 */

package org.esupportail.portal.utils.database;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Classe d'abstraction pour accs aux base de donnes.
 * L'objectif est de simplifier les accs les plus courants  la librairie JDBC et 
 * de centraliser les exceptions qu'elles peuvent soulever.
 */
public class Query {
	
	protected static Log logguer = LogFactory.getLog(Query.class);
    
    private Connection con = null;
	private PreparedStatement statement = null;
	private ResultSet resultset = null;
	private static final String PREFIX = "__";
	private boolean autoCommit = true;
 	private boolean transactionStarted = false;
	private int isolation = Connection.TRANSACTION_READ_COMMITTED;
	
	/**
	 * Constructeur 
	 * @param db objet database  utiliser pour la requte
	 */
	public Query(Database db) throws SQLException {
		if (db != null) {
			setCon(db);
		}
		else {
			logguer.error("Query::Query() : Objet Database vide");
			throw new SQLException();
		}	
	}
	
	/**
	 * Fermeture de la requete : resultset, statement et connexion
	 */
	public Exception close() {
		Exception res = null;
		try {
			closeRs();
		}
		catch (Exception e) {
				res = e;
				logException(e, "close resultset");
		}
		try {
			closeStmt();
		}
		catch (Exception e) {
			res = e;
			logException(e, "close statement");
		}
		try {
			if (getCon() != null) {
				getCon().close();
			}
		}
		catch (Exception e) {
			res = e;
			logException(e, "close connection");
		}
		return res;
	}
	
	/**
	 * Fermeture du resultset associ  l'objet
	 * @throws SQLException
	 */
	private void closeRs() throws SQLException {
		if (getRs() != null) {
			getRs().close();
		}
	}
	
	/**
	 * Fermeture du statement associ  l'objet
	 * @throws SQLException
	 */
	private void closeStmt() throws SQLException {
		if (getStmt() != null) {
			getStmt().close();
		}
	}
	/**
	 * rcupre la connexion associe  l'objet
	 */
	private Connection getCon() {
		return con;
	}

	/**
	 * rcupre le resultset associ  l'objet
	 */
	public ResultSet getRs() {
		return resultset;
	}

	/**
	 * retourne le statement associ  l'objet
	 */
	public PreparedStatement getStmt() {
		return statement;
	}

	/**
	 * Log les exceptions
	 * 
	 * @param exception
	 * @param comment commentaire  afficher dans le log
	 */
	private void logException(Exception exception, String comment) {
		logguer.error("Query:logExeception() : " + comment + "(" + exception.getMessage() + ")");
	}

	/**
	 * Positionnement de la requte SQL de l'objet
	 * @param sql
	 * @throws SQLException
	 */
	public void setSql(String sql) throws SQLException {
		// Fermeture ventuelle de la requte en cours
		closeRs();
		closeStmt();
		// Puis ouverture
		setStmt(getCon().prepareStatement(sql));
		statement.clearParameters();
/*		
		// temporaire OZI
	    Context initCtx;
		try {
			initCtx = new InitialContext();
		    Context envCtx = (Context) initCtx.lookup("java:comp/env");
			BasicDataSource ds = (BasicDataSource)envCtx.lookup("jdbc/Apogee");
			logguer.debug("Nombre de connexions : " + String.valueOf(ds.getNumActive()) + " - " + sql);
		} catch (NamingException e) {
			logguer.error(e);
		}
*/
	}
	
	/**
	 * Positionne le sql en positionnant un prfixe devant les tables de la requte
	 * @param sql requte SQL
	 * @param prefix prfixe  ajouter en lieu et place de __
	 * @throws SQLException
	 */
	public void setSql(String sql, String prefix) throws SQLException {
		setSql(sql.replaceAll(PREFIX,prefix));
	}

	/**
	 * Ouverture de la requte
	 * @throws SQLException
	 */
	public void select() throws SQLException {
		if (getStmt() == null)
			throw new SQLException("select : le statement est null");
		setRs(getStmt().executeQuery());
	}

	/**
	 * Commit
	 */
	public void commit() {
		try {
			con.commit();	
		}
		catch (SQLException e) {
		    logException(e, "commit");
		}
	}
	
	/**
	 * rollback
	 */
	public void rollback() {
		try {
			con.rollback();	
		}
		catch (SQLException e) {
		    logException(e, "rollback");
		}
	}
	
	/**
	 * Execution d'un ordre SQLs
	 * @param mode : insert/delete/update (ne sert que pour les logs)
	 * @return
	 * @throws SQLException
	 */
	private int execSql(String mode) throws SQLException {
		if(!autoCommit && !transactionStarted) {
		    transactionStarted = true;
			con.setAutoCommit(false);
			con.setTransactionIsolation(isolation);
		}
		if (getStmt() == null)
			throw new SQLException(mode + " : le statement est null");
		int res = (getStmt().executeUpdate());
		return res;
	}
	
	/**
	 * Insertion
	 * @throws SQLException
	 */
	public int insert() throws SQLException {
		return execSql("insert");
	}
	
	/**
	 * Delete
	 * @throws SQLException
	 */
	public int delete() throws SQLException {
		return execSql("delete");
	}
	
	/**
	 * Update
	 * @throws SQLException
	 */
	public int update() throws SQLException {
		return execSql("update");
	}
	
	/**
	 * Positionnement de la requte et ouverture de celle-ci en une opration
	 * @param sql requte sql
	 * @throws SQLException
	 */
	public void select(String sql) throws SQLException {
		setSql(sql);
		select();
	}
	
	/**
	 * Postionnement de la connexion de l'objet
	 * @param db objet Database
	 */
	private void setCon(Database db) {
		// cas pool Tomcat			
		if (db.getType().toUpperCase().equals("JNDI")) {
			try {
				Context initCtx = new InitialContext();
			    Context envCtx = (Context) initCtx.lookup("java:comp/env");
			    DataSource ds = (DataSource)envCtx.lookup("jdbc/" + db.getUrl());
			    if(ds != null) {
			    	con = ds.getConnection();
			    }
			    else {
			    	logguer.error("Query::setCon() : La base '" + db.getUrl() + "' est introuvable");
			    }
			}
			catch(javax.naming.NamingException ne) {
				logguer.error("Query::setCon() : Exception lors de la recherche du pool JNDI [" + db.getUrl() + "]", ne);
			}
			catch(SQLException sqle) {
				logguer.error("Query::setCon() : Exception lors de la connexion au pool JNDI [" + db.getUrl() + "]", sqle);
			}
			if (con == null) {
				logguer.error("Query::setCon(Database) : connexion vide");
			}
		} // cas JDBC simple
		else if (db.getType().toUpperCase().equals("JDBC")) {
			createConnection(db);
			// Si l'objet connexion n'est pas null mais qu'il n'est plus valide, on le recre		
			try{
				if(con.isClosed()){
					createConnection(db);
				}
			}
			catch(SQLException e){
			    logguer.error("Query::setCon() Erreur de connexion \u00E0 l'url : " + db.getUrl() + ";" + e);
			}				
		}
		else { // pas de type prcis
		    logguer.error("Query::setCon() Erreur de connexion \u00E0 l'url : " + db.getUrl() + "; Pas de type de connexion pr\u00E9cis\u00E9");
		}
	}
	
	/**
	 * Cration d'une connexion JDBC 
	 * @param db objet database
	 */
	private void createConnection(Database db){
		try{
			Class.forName(db.getDriverClassName()).newInstance();
			con = DriverManager.getConnection(db.getUrl(),db.getUsername(),db.getPassword());
			if (con == null){
			    logguer.error("Query::createConnection() Erreur de connexion \u00E0 l'url : " + db.getUrl());
			}
		}
		catch(Exception e){
		    logguer.error("Query::createConnection() Erreur de connexion \u00E0 l'url : " + db.getUrl() + ";" + e);
		}
	}

	/**
	 * @param set
	 */
	private void setRs(ResultSet resultset) {
		this.resultset = resultset;
	}

	/**
	 * @param statement
	 */
	private void setStmt(PreparedStatement statement) {
		this.statement = statement;
	}
	
	/**
	 * l'autocommit est-il actif ?
	 * @return
	 */
	public boolean isAutoCommit() {
		return autoCommit;
	}

	/**
	 * Positionnement de l'autocommit
	 * @param autocommit
	 */
	public void setAutoCommit(boolean autocommit) {
		this.autoCommit = autocommit; 
	}

	/**
	 * Retourne une chaine entre '
	 * @param s chaine  mettre entre '
	 * @return
	 */
	public static String quotedStr(String s) {
		return "'" + s + "'";
	}
	
	public void setTransactionIsolation(int level) {
		isolation = level;
	}
}